//=============================================================================
// OpenTypePixelFont.js
//=============================================================================

/*:
 * @plugindesc Draw true type fonts using opentype.js
 * @author utunnels
 *
 * @param fontList
 * @type struct<OpenTypeFont>[]
 * @text font list
 * @desc Define fonts you want to use in game.
 * @default ["{\"family\":\"GameFont\",\"file\":\"PixelOperator.ttf\",\"snapToPixel\":\"true\",\"baseSize\":\"16\",\"unicode-range\":\"U+0-10FFFF\"}","{\"family\":\"rmmz-mainfont\",\"file\":\"PixelOperator.ttf\",\"snapToPixel\":\"true\",\"baseSize\":\"16\",\"unicode-range\":\"U+0-10FFFF\"}","{\"family\":\"rmmz-numberfont\",\"file\":\"PixelOperator8.ttf\",\"snapToPixel\":\"true\",\"baseSize\":\"{20:16,34:24,30:24}\",\"unicode-range\":\"U+0-10FFFF\"}"]
 *
 * @help
 *
Font list format:

font# file
File name of the font file, for example pixeloperator.ttf

font# family name
This is the font name, for example, if you want to replace gamefont, 
you name it gamefont. For MZ, the default font is rmmz-mainfont and 
rmmz-numberfont, the later is used to display gauge values.
However, this plugin doesn't provide any method to change in-game fonts. 
You'll need to do that by yourself.

font# snap to pixel
Whether this font needs to snap to pixel grid. 
This is important if you don't want the strokes to 'squat' between pixels.
Usually this is for pixel fonts as the name suggested.

font# base size
The base size of the pixel font. Usually it means the size for the font to 
display clear single pixel strokes.
For example, it is 16 for pixeloperator, 8 for pixeloperator8.
If your game draws text with font size of 28, and base size is 16, 
the closest whole number scale will be used, so 32 will be used. 
This only affects how the text is displayed, it will not change 
in game variables.
You can also provide a map for different sizes, 
for example: {28:24, 72:80}, it means if the font size is 28, 
the plugin will use 24, and if the font size is 72, the plugin will use 80.

font# unicode range
Unicdoe range of the font. It follows the same rules of css unicode-range.
The default value is U+0-10FFFF, you only need to change it if you use 
multiple font files for the same font family, to specify which characters 
font# need to display. 
 */
/*:zh
 * @plugindesc opentype像素字体绘制
 * @author utunnels
 *
 * @param fontList
 * @type struct<OpenTypeFont>[]
 * @text 字体列表
 * @desc 定义游戏中用到的字体
 * @default ["{\"family\":\"GameFont\",\"file\":\"PixelOperator.ttf\",\"snapToPixel\":\"true\",\"baseSize\":\"16\",\"unicode-range\":\"U+0-10FFFF\"}","{\"family\":\"rmmz-mainfont\",\"file\":\"PixelOperator.ttf\",\"snapToPixel\":\"true\",\"baseSize\":\"16\",\"unicode-range\":\"U+0-10FFFF\"}","{\"family\":\"rmmz-numberfont\",\"file\":\"PixelOperator8.ttf\",\"snapToPixel\":\"true\",\"baseSize\":\"{20:16,34:24,30:24}\",\"unicode-range\":\"U+0-10FFFF\"}"]
 *
 * @help
 *
文字列表格式:

字体文件
fonts目录下的字体文件命，例如pixeloperator.ttf

字体名称
比如，你想要替换掉GameFont，就把字体命名为GameFont。
注意，本插件不提供绘制文字之外的功能，如果你需要更改游戏中的字体，
你需要其它的插件，在这里，只要把需要替换的字体名称写上。
默认情况下，MV的字体是GameFont，MZ是rmmz-mainfont和rmmz-numberfont，

对齐像素
是否对齐像素，对于像素字体，应该保留默认值ON(true)。

基本字号
字体的基本大小，就是字体绘制出来保持像素字外观的最小尺寸。
例如范例里面pixeloperator.ttf是16，pixeloperator8.ttf是8.
实际绘制文字的时候会采用这个字号的整数倍，例如，如果基本字号是16，
游戏中用到28号字会采用最接近的32号字。
如果有条件，最好是修改文字的实际尺寸，本插件不提供相关功能。
字号也可以分别设置，例如{28:24, 72:80}，这样28号字直接对应24号，
72号对应80号，如果你需要微调，可以给每一个可能的大小分别指定字号。

unicode范围
顾名思义就是这个字体的unicode范围，具体规则和css的unicode-range一致。
默认值是U+0-10FFFF，如果你有多个同名字体需要分别设置unicode范围，
你可能需要用到这个功能。例如你想让一个字体显示英文部分，另一个字体
显示中文部分。如果用不到就保留默认值（全部字符）。
 */
/*:ja
 * @plugindesc opentypeドットフォントレンダラー
 * @author utunnels
 *
 * @param fontList
 * @type struct<OpenTypeFont>[]
 * @text フォントリスト
 * @desc ゲームで使用するフォントを定義する。
 * @default ["{\"family\":\"GameFont\",\"file\":\"PixelOperator.ttf\",\"snapToPixel\":\"true\",\"baseSize\":\"16\",\"unicode-range\":\"U+0-10FFFF\"}","{\"family\":\"rmmz-mainfont\",\"file\":\"PixelOperator.ttf\",\"snapToPixel\":\"true\",\"baseSize\":\"16\",\"unicode-range\":\"U+0-10FFFF\"}","{\"family\":\"rmmz-numberfont\",\"file\":\"PixelOperator8.ttf\",\"snapToPixel\":\"true\",\"baseSize\":\"{20:16,34:24,30:24}\",\"unicode-range\":\"U+0-10FFFF\"}"]
 *
 * @help
 *
フォントリスト形式:

フォントファイル
fonts/フォルダ内のフォントファイル名、例えばpixeloperator.ttf。

フォントファミリー
例えば、フォント「GameFont」を置き換えたい場合、
「GameFont」という名前をつけます。
デフォルトでは、MVのフォントは「GameFont」、
MZのフォントは「rmmz-mainfont」と「rmmz-numberfont」です。

ピクセルにスナップ
フォントのドットを最も近いピクセルにスナップするかどうか。

基本フォントサイズ
これは通常、フォントがはっきりと表示されるための最小サイズを意味する。
たとえば、pixeloperator.ttfは16で、pixeloperator8.ttfは8です。
ゲームのテキスト描画サイズが28の場合、基本サイズは16で、32が使用されます。
また、個々のフォントサイズを定義することもできます。例えば、{28:24, 72:80}

unicode-range
これはcssのunicode-rangeと同じである。
デフォルト値はU+0-10FFFF、にはすべての可能な文字が含まれている。
複数の異なるフォントを使って異なる範囲の文字を表示したい場合は、
unicode-rangeを編集することができる。
 */
 /*~struct~OpenTypeFont:zh
  * @param family
  * @text 字体名称
  * @default GameFont
  * @desc 游戏中用到的字体名称，详见说明。
  *
  * @param file
  * @text 字体文件带扩展名
  * @default PixelOperator.ttf
  * @desc fonts下的字体文件名称，包含扩展名
  *
  * @param snapToPixel
  * @text 对齐像素
  * @type boolean
  * @default true
  * @desc 是否对齐像素，防止文字模糊。
  *
  * @param baseSize
  * @text 基本字号
  * @default 16
  * @desc 最小像素尺寸，详见说明。
  * 
  * @param unicode-range
  * @text unicode范围
  * @default U+0-10FFFF
  * @desc unicode编码范围，规则参考：https://developer.mozilla.org/zh-CN/docs/Web/CSS/%40font-face/unicode-range
  */
 /*~struct~OpenTypeFont:en
  * @param family
  * @text font family name
  * @default GameFont
  * @desc This is the font name you use in game, for MV the default value should be GameFont, for MZ you should use rmmz-manfont.
  *
  * @param file
  * @text file name with extension
  * @default PixelOperator.ttf
  * @desc This is the file name of the font file you want to use. Files should be placed in fonts/ folder.
  *
  * @param snapToPixel
  * @text snap to pixel
  * @type boolean
  * @default true
  * @desc Whether this font snaps to pixel grids, to prevent blurring.
  *
  * @param baseSize
  * @text base size
  * @default 16
  * @desc Base size of the font, only whole number scales are possible. Check plugin help for examples.
  * 
  * @param unicode-range
  * @default U+0-10FFFF
  * @desc Unicode range of the font, use the default value if you don't know what to do with it. For details check https://developer.mozilla.org/en-US/docs/Web/CSS/%40font-face/unicode-range
  */
 /*~struct~OpenTypeFont:ja
  * @param family
  * @text フォントファミリー
  * @default GameFont
  * @desc ゲーム内で使用したフォントの名前です。
  *
  * @param file
  * @text フォントファイル
  * @default PixelOperator.ttf
  * @desc fonts/フォルダ内のフォントファイル名、例えばpixeloperator.ttf。
  *
  * @param snapToPixel
  * @text ピクセルにスナップ
  * @type boolean
  * @default true
  * @desc フォントのドットを最も近いピクセルにスナップするかどうか。
  *
  * @param baseSize
  * @text 基本フォントサイズ
  * @default 16
  * @desc これはフォントの基本サイズです。テキストはこのサイズの倍数でのみ表示されます。
  * 
  * @param unicode-range
  * @default U+0-10FFFF
  * @desc これはcssのunicode-rangeと同じである。参照：https://developer.mozilla.org/ja-JP/docs/Web/CSS/%40font-face/unicode-range
  */

(function(){
///////////////////////////////opentype.js//////////////////////////////////////////
/*
The MIT License (MIT)

Copyright (c) 2020 Frederik De Bleser

Permission is hereby granted, free of charge, to any person obtaining a copy of
this software and associated documentation files (the "Software"), to deal in
the Software without restriction, including without limitation the rights to
use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
the Software, and to permit persons to whom the Software is furnished to do so,
subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/
var opentype=(()=>{var e=Object.defineProperty,t=Object.getOwnPropertyDescriptor,n=Object.getOwnPropertyNames,r=Object.prototype.hasOwnProperty,s={};((t,n)=>{for(var r in n)e(t,r,{get:n[r],enumerable:!0})})(s,{BoundingBox:()=>L,Font:()=>io,Glyph:()=>yt,Path:()=>F,_parse:()=>Ee,load:()=>Uo,loadSync:()=>Ro,parse:()=>Oo});var o=0,a=-3;function i(){this.table=new Uint16Array(16),this.trans=new Uint16Array(288)}function l(e,t){this.source=e,this.sourceIndex=0,this.tag=0,this.bitcount=0,this.dest=t,this.destLen=0,this.ltree=new i,this.dtree=new i}var u=new i,c=new i,p=new Uint8Array(30),h=new Uint16Array(30),f=new Uint8Array(30),d=new Uint16Array(30),g=new Uint8Array([16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15]),m=new i,y=new Uint8Array(320);function v(e,t,n,r){var s,o;for(s=0;s<n;++s)e[s]=0;for(s=0;s<30-n;++s)e[s+n]=s/n|0;for(o=r,s=0;s<30;++s)t[s]=o,o+=1<<e[s]}var b=new Uint16Array(16);function S(e,t,n,r){var s,o;for(s=0;s<16;++s)e.table[s]=0;for(s=0;s<r;++s)e.table[t[n+s]]++;for(e.table[0]=0,o=0,s=0;s<16;++s)b[s]=o,o+=e.table[s];for(s=0;s<r;++s)t[n+s]&&(e.trans[b[t[n+s]]++]=s)}function x(e){e.bitcount--||(e.tag=e.source[e.sourceIndex++],e.bitcount=7);var t=1&e.tag;return e.tag>>>=1,t}function T(e,t,n){if(!t)return n;for(;e.bitcount<24;)e.tag|=e.source[e.sourceIndex++]<<e.bitcount,e.bitcount+=8;var r=e.tag&65535>>>16-t;return e.tag>>>=t,e.bitcount-=t,r+n}function k(e,t){for(;e.bitcount<24;)e.tag|=e.source[e.sourceIndex++]<<e.bitcount,e.bitcount+=8;var n=0,r=0,s=0,o=e.tag;do{r=2*r+(1&o),o>>>=1,++s,n+=t.table[s],r-=t.table[s]}while(r>=0);return e.tag=o,e.bitcount-=s,t.trans[n+r]}function O(e,t,n){var r,s,o,a,i,l;for(r=T(e,5,257),s=T(e,5,1),o=T(e,4,4),a=0;a<19;++a)y[a]=0;for(a=0;a<o;++a){var u=T(e,3,0);y[g[a]]=u}for(S(m,y,0,19),i=0;i<r+s;){var c=k(e,m);switch(c){case 16:var p=y[i-1];for(l=T(e,2,3);l;--l)y[i++]=p;break;case 17:for(l=T(e,3,3);l;--l)y[i++]=0;break;case 18:for(l=T(e,7,11);l;--l)y[i++]=0;break;default:y[i++]=c}}S(t,y,0,r),S(n,y,r,s)}function U(e,t,n){for(;;){var r,s,a,i,l=k(e,t);if(256===l)return o;if(l<256)e.dest[e.destLen++]=l;else for(r=T(e,p[l-=257],h[l]),s=k(e,n),i=a=e.destLen-T(e,f[s],d[s]);i<a+r;++i)e.dest[e.destLen++]=e.dest[i]}}function R(e){for(var t,n;e.bitcount>8;)e.sourceIndex--,e.bitcount-=8;if((t=256*(t=e.source[e.sourceIndex+1])+e.source[e.sourceIndex])!==(65535&~(256*e.source[e.sourceIndex+3]+e.source[e.sourceIndex+2])))return a;for(e.sourceIndex+=4,n=t;n;--n)e.dest[e.destLen++]=e.source[e.sourceIndex++];return e.bitcount=0,o}function w(e,t){var n,r,s=new l(e,t);do{switch(n=x(s),T(s,2,0)){case 0:r=R(s);break;case 1:r=U(s,u,c);break;case 2:O(s,s.ltree,s.dtree),r=U(s,s.ltree,s.dtree);break;default:r=a}if(r!==o)throw new Error("Data error")}while(!n);return s.destLen<s.dest.length?"function"==typeof s.dest.slice?s.dest.slice(0,s.destLen):s.dest.subarray(0,s.destLen):s.dest}function I(e,t,n,r,s){return Math.pow(1-s,3)*e+3*Math.pow(1-s,2)*s*t+3*(1-s)*Math.pow(s,2)*n+Math.pow(s,3)*r}function C(){this.x1=Number.NaN,this.y1=Number.NaN,this.x2=Number.NaN,this.y2=Number.NaN}!function(e,t){var n;for(n=0;n<7;++n)e.table[n]=0;for(e.table[7]=24,e.table[8]=152,e.table[9]=112,n=0;n<24;++n)e.trans[n]=256+n;for(n=0;n<144;++n)e.trans[24+n]=n;for(n=0;n<8;++n)e.trans[168+n]=280+n;for(n=0;n<112;++n)e.trans[176+n]=144+n;for(n=0;n<5;++n)t.table[n]=0;for(t.table[5]=32,n=0;n<32;++n)t.trans[n]=n}(u,c),v(p,h,4,3),v(f,d,2,1),p[28]=0,h[28]=258,C.prototype.isEmpty=function(){return isNaN(this.x1)||isNaN(this.y1)||isNaN(this.x2)||isNaN(this.y2)},C.prototype.addPoint=function(e,t){"number"==typeof e&&((isNaN(this.x1)||isNaN(this.x2))&&(this.x1=e,this.x2=e),e<this.x1&&(this.x1=e),e>this.x2&&(this.x2=e)),"number"==typeof t&&((isNaN(this.y1)||isNaN(this.y2))&&(this.y1=t,this.y2=t),t<this.y1&&(this.y1=t),t>this.y2&&(this.y2=t))},C.prototype.addX=function(e){this.addPoint(e,null)},C.prototype.addY=function(e){this.addPoint(null,e)},C.prototype.addBezier=function(e,t,n,r,s,o,a,i){const l=[e,t],u=[n,r],c=[s,o],p=[a,i];this.addPoint(e,t),this.addPoint(a,i);for(let e=0;e<=1;e++){const t=6*l[e]-12*u[e]+6*c[e],n=-3*l[e]+9*u[e]-9*c[e]+3*p[e],r=3*u[e]-3*l[e];if(0===n){if(0===t)continue;const n=-r/t;0<n&&n<1&&(0===e&&this.addX(I(l[e],u[e],c[e],p[e],n)),1===e&&this.addY(I(l[e],u[e],c[e],p[e],n)));continue}const s=Math.pow(t,2)-4*r*n;if(s<0)continue;const o=(-t+Math.sqrt(s))/(2*n);0<o&&o<1&&(0===e&&this.addX(I(l[e],u[e],c[e],p[e],o)),1===e&&this.addY(I(l[e],u[e],c[e],p[e],o)));const a=(-t-Math.sqrt(s))/(2*n);0<a&&a<1&&(0===e&&this.addX(I(l[e],u[e],c[e],p[e],a)),1===e&&this.addY(I(l[e],u[e],c[e],p[e],a)))}},C.prototype.addQuad=function(e,t,n,r,s,o){const a=e+2/3*(n-e),i=t+2/3*(r-t),l=a+1/3*(s-e),u=i+1/3*(o-t);this.addBezier(e,t,a,i,l,u,s,o)};var L=C;function E(){this.commands=[],this.fill="black",this.stroke=null,this.strokeWidth=1}var D={};function M(e,t){const n=Math.floor(e),r=e-n;if(D[t]||(D[t]={}),void 0!==D[t][r]){return n+D[t][r]}const s=+(Math.round(r+"e+"+t)+"e-"+t);return D[t][r]=s,n+s}function A(e){let t=[[]],n=0,r=0;for(let s=0;s<e.length;s+=1){const o=t[t.length-1],a=e[s],i=o[0],l=o[1],u=o[o.length-1],c=e[s+1];o.push(a),"M"===a.type?(n=a.x,r=a.y):"L"!==a.type||c&&"Z"!==c.command?"L"===a.type&&u&&u.x===a.x&&u.y===a.y?o.pop():"Z"===a.type&&(i&&l&&u&&"M"===i.type&&"L"===l.type&&"L"===u.type&&u.x===i.x&&u.y===i.y&&(o.shift(),o[0].type="M"),s+1<e.length&&t.push([])):Math.abs(a.x-n)>1||Math.abs(a.y-r)>1||o.pop()}return e=[].concat.apply([],t)}E.prototype.fromSVG=function(e,t={}){"undefined"!=typeof SVGPathElement&&e instanceof SVGPathElement&&(e=e.getAttribute("d")),t=function(e){return Object.assign({},{decimalPlaces:2,optimize:!0,flipY:!0,flipYBase:void 0,scale:1,x:0,y:0},e)}(t),this.commands=[];const n="MmLlQqCcZzHhVv";let r={},s=[""],o=!1;function a(e){if(!this.commands.length)return e;const t=this.commands[this.commands.length-1];for(let n=0;n<e.length;n++)e[n]+=t[1&n?"y":"x"];return e}function i(){if(void 0===r.type)return;const e=r.type.toUpperCase(),n="Z"!==e&&r.type.toUpperCase()!==r.type;let o=s.filter((e=>e.length)).map((e=>{let n=parseFloat(e);return(t.decimalPlaces||0===t.decimalPlaces)&&(n=M(n,t.decimalPlaces)),n}));if(s=[""],!o.length&&"Z"!==e)return;n&&"H"!==e&&"V"!==e&&(o=a.apply(this,[o]));const i=this.commands.length&&this.commands[this.commands.length-1].x||0,l=this.commands.length&&this.commands[this.commands.length-1].y||0;switch(e){case"M":this.moveTo(...o);break;case"L":this.lineTo(...o);break;case"V":for(let e=0;e<o.length;e++){let t=0;n&&(t=this.commands.length&&this.commands[this.commands.length-1].y||0),this.lineTo(i,o[e]+t)}break;case"H":for(let e=0;e<o.length;e++){let t=0;n&&(t=this.commands.length&&this.commands[this.commands.length-1].x||0),this.lineTo(o[e]+t,l)}break;case"C":this.bezierCurveTo(...o);break;case"Q":this.quadraticCurveTo(...o);break;case"Z":(this.commands.length<1||"Z"!==this.commands[this.commands.length-1].type)&&this.close()}if(this.commands.length)for(const e in this.commands[this.commands.length-1])void 0===this.commands[this.commands.length-1][e]&&(this.commands[this.commands.length-1][e]=0)}for(let t=0;t<e.length;t++){const a=e.charAt(t),l=s[s.length-1];if("0123456789".indexOf(a)>-1)s[s.length-1]+=a;else if("-+".indexOf(a)>-1)if(r.type||this.commands.length||(r.type="L"),"-"===a)!r.type||l.indexOf("-")>0?o=!0:l.length?s.push("-"):s[s.length-1]=a;else{if(r.type&&!(l.length>0))continue;o=!0}else if(n.indexOf(a)>-1)r.type?(i.apply(this),r={type:a}):r.type=a;else{if("SsTtAa".indexOf(a)>-1)throw new Error("Unsupported path command: "+a+". Currently supported commands are "+n.split("").join(", ")+".");" ,\t\n\r\f\v".indexOf(a)>-1?s.push(""):"."===a?!r.type||l.indexOf(a)>-1?o=!0:s[s.length-1]+=a:o=!0}if(o)throw new Error("Unexpected character: "+a+" at offset "+t)}i.apply(this),t.optimize&&(this.commands=A(this.commands));const l=t.flipY;let u=t.flipYBase;if(!0===l&&void 0===t.flipYBase){const e=this.getBoundingBox();u=e.y1+e.y2}for(const e in this.commands){const n=this.commands[e];for(const r in n)["x","x1","x2"].includes(r)?this.commands[e][r]=t.x+n[r]*t.scale:["y","y1","y2"].includes(r)&&(this.commands[e][r]=t.y+(l?u-n[r]:n[r])*t.scale)}return this},E.fromSVG=function(e,t){return(new E).fromSVG(e,t)},E.prototype.moveTo=function(e,t){this.commands.push({type:"M",x:e,y:t})},E.prototype.lineTo=function(e,t){this.commands.push({type:"L",x:e,y:t})},E.prototype.curveTo=E.prototype.bezierCurveTo=function(e,t,n,r,s,o){this.commands.push({type:"C",x1:e,y1:t,x2:n,y2:r,x:s,y:o})},E.prototype.quadTo=E.prototype.quadraticCurveTo=function(e,t,n,r){this.commands.push({type:"Q",x1:e,y1:t,x:n,y:r})},E.prototype.close=E.prototype.closePath=function(){this.commands.push({type:"Z"})},E.prototype.extend=function(e){if(e.commands)e=e.commands;else if(e instanceof L){const t=e;return this.moveTo(t.x1,t.y1),this.lineTo(t.x2,t.y1),this.lineTo(t.x2,t.y2),this.lineTo(t.x1,t.y2),void this.close()}Array.prototype.push.apply(this.commands,e)},E.prototype.getBoundingBox=function(){const e=new L;let t=0,n=0,r=0,s=0;for(let o=0;o<this.commands.length;o++){const a=this.commands[o];switch(a.type){case"M":e.addPoint(a.x,a.y),t=r=a.x,n=s=a.y;break;case"L":e.addPoint(a.x,a.y),r=a.x,s=a.y;break;case"Q":e.addQuad(r,s,a.x1,a.y1,a.x,a.y),r=a.x,s=a.y;break;case"C":e.addBezier(r,s,a.x1,a.y1,a.x2,a.y2,a.x,a.y),r=a.x,s=a.y;break;case"Z":r=t,s=n;break;default:throw new Error("Unexpected path command "+a.type)}}return e.isEmpty()&&e.addPoint(0,0),e},E.prototype.draw=function(e){const t=this._layers;if(t&&t.length){for(let n=0;n<t.length;n++)this.draw.call(t[n],e);return}const n=this._image;if(n)e.drawImage(n.image,n.x,n.y,n.width,n.height);else{e.beginPath();for(let t=0;t<this.commands.length;t+=1){const n=this.commands[t];"M"===n.type?(e.closePath(),e.moveTo(n.x,n.y)):"L"===n.type?e.lineTo(n.x,n.y):"C"===n.type?e.bezierCurveTo(n.x1,n.y1,n.x2,n.y2,n.x,n.y):"Q"===n.type?e.quadraticCurveTo(n.x1,n.y1,n.x,n.y):"Z"===n.type&&this.stroke&&this.strokeWidth&&e.closePath()}e.closePath(),this.fill&&(e.fillStyle=this.fill,e.fill()),this.stroke&&(e.strokeStyle=this.stroke,e.lineWidth=this.strokeWidth,e.stroke())}},E.prototype.toPathData=function(e){function t(t){const n=M(t,e.decimalPlaces);return Math.round(t)===n?""+n:n.toFixed(e.decimalPlaces)}function n(){let e="";for(let n=0;n<arguments.length;n+=1){const r=arguments[n];r>=0&&n>0&&(e+=" "),e+=t(r)}return e}e=function(e){return parseInt(e)===e&&(e={decimalPlaces:e,flipY:!1}),Object.assign({},{decimalPlaces:2,optimize:!0,flipY:!0,flipYBase:void 0},e)}(e);let r=this.commands;e.optimize&&(r=JSON.parse(JSON.stringify(this.commands)),r=A(r));const s=e.flipY;let o=e.flipYBase;if(!0===s&&void 0===o){const e=new E;e.extend(r);const t=e.getBoundingBox();o=t.y1+t.y2}let a="";for(let e=0;e<r.length;e+=1){const t=r[e];"M"===t.type?a+="M"+n(t.x,s?o-t.y:t.y):"L"===t.type?a+="L"+n(t.x,s?o-t.y:t.y):"C"===t.type?a+="C"+n(t.x1,s?o-t.y1:t.y1,t.x2,s?o-t.y2:t.y2,t.x,s?o-t.y:t.y):"Q"===t.type?a+="Q"+n(t.x1,s?o-t.y1:t.y1,t.x,s?o-t.y:t.y):"Z"===t.type&&(a+="Z")}return a},E.prototype.toSVG=function(e,t){this._layers&&this._layers.length&&console.warn("toSVG() does not support colr font layers yet"),this._image&&console.warn("toSVG() does not support SVG glyphs yet"),t||(t=this.toPathData(e));let n='<path d="';return n+=t,n+='"',void 0!==this.fill&&"black"!==this.fill&&(null===this.fill?n+=' fill="none"':n+=' fill="'+this.fill+'"'),this.stroke&&(n+=' stroke="'+this.stroke+'" stroke-width="'+this.strokeWidth+'"'),n+="/>",n},E.prototype.toDOMElement=function(e,t){this._layers&&this._layers.length&&console.warn("toDOMElement() does not support colr font layers yet"),t||(t=this.toPathData(e));const n=t,r=document.createElementNS("http://www.w3.org/2000/svg","path");return r.setAttribute("d",n),void 0!==this.fill&&"black"!==this.fill&&(null===this.fill?r.setAttribute("fill","none"):r.setAttribute("fill",this.fill)),this.stroke&&(r.setAttribute("stroke",this.stroke),r.setAttribute("stroke-width",this.strokeWidth)),r};var F=E;function P(e){throw new Error(e)}function N(e,t){e||P(t)}var G={fail:P,argument:N,assert:N},B=2147483648,H={},z={},V={};function q(e){return function(){return e}}z.BYTE=function(e){return G.argument(e>=0&&e<=255,"Byte value should be between 0 and 255."),[e]},V.BYTE=q(1),z.CHAR=function(e){return[e.charCodeAt(0)]},V.CHAR=q(1),z.CHARARRAY=function(e){null==e&&(e="",console.warn("CHARARRAY with undefined or null value encountered and treated as an empty string. This is probably caused by a missing glyph name."));const t=[];for(let n=0;n<e.length;n+=1)t[n]=e.charCodeAt(n);return t},V.CHARARRAY=function(e){return void 0===e?0:e.length},z.USHORT=function(e){return[e>>8&255,255&e]},V.USHORT=q(2),z.SHORT=function(e){return e>=32768&&(e=-(65536-e)),[e>>8&255,255&e]},V.SHORT=q(2),z.UINT24=function(e){return[e>>16&255,e>>8&255,255&e]},V.UINT24=q(3),z.ULONG=function(e){return[e>>24&255,e>>16&255,e>>8&255,255&e]},V.ULONG=q(4),z.LONG=function(e){return e>=B&&(e=-(2*B-e)),[e>>24&255,e>>16&255,e>>8&255,255&e]},V.LONG=q(4),z.FLOAT=function(e){if(e>32767.00001525879||e<-32768)throw new Error(`Value ${e} is outside the range of representable values in 16.16 format`);const t=Math.round(65536*e)<<0;return z.ULONG(t)},V.FLOAT=V.ULONG,z.FIXED=z.ULONG,V.FIXED=V.ULONG,z.FWORD=z.SHORT,V.FWORD=V.SHORT,z.UFWORD=z.USHORT,V.UFWORD=V.USHORT,z.F2DOT14=function(e){return z.USHORT(16384*e)},V.F2DOT14=V.USHORT,z.LONGDATETIME=function(e){return[0,0,0,0,e>>24&255,e>>16&255,e>>8&255,255&e]},V.LONGDATETIME=q(8),z.TAG=function(e){return G.argument(4===e.length,"Tag should be exactly 4 ASCII characters."),[e.charCodeAt(0),e.charCodeAt(1),e.charCodeAt(2),e.charCodeAt(3)]},V.TAG=q(4),z.Card8=z.BYTE,V.Card8=V.BYTE,z.Card16=z.USHORT,V.Card16=V.USHORT,z.OffSize=z.BYTE,V.OffSize=V.BYTE,z.SID=z.USHORT,V.SID=V.USHORT,z.NUMBER=function(e){return e>=-107&&e<=107?[e+139]:e>=108&&e<=1131?[247+((e-=108)>>8),255&e]:e>=-1131&&e<=-108?[251+((e=-e-108)>>8),255&e]:e>=-32768&&e<=32767?z.NUMBER16(e):z.NUMBER32(e)},V.NUMBER=function(e){return z.NUMBER(e).length},z.NUMBER16=function(e){return[28,e>>8&255,255&e]},V.NUMBER16=q(3),z.NUMBER32=function(e){return[29,e>>24&255,e>>16&255,e>>8&255,255&e]},V.NUMBER32=q(5),z.REAL=function(e){let t=e.toString();const n=/\.(\d*?)(?:9{5,20}|0{5,20})\d{0,2}(?:e(.+)|$)/.exec(t);if(n){const r=parseFloat("1e"+((n[2]?+n[2]:0)+n[1].length));t=(Math.round(e*r)/r).toString()}let r="";for(let e=0,n=t.length;e<n;e+=1){const n=t[e];r+="e"===n?"-"===t[++e]?"c":"b":"."===n?"a":"-"===n?"e":n}r+=1&r.length?"f":"ff";const s=[30];for(let e=0,t=r.length;e<t;e+=2)s.push(parseInt(r.substr(e,2),16));return s},V.REAL=function(e){return z.REAL(e).length},z.NAME=z.CHARARRAY,V.NAME=V.CHARARRAY,z.STRING=z.CHARARRAY,V.STRING=V.CHARARRAY,H.UTF8=function(e,t,n){const r=[],s=n;for(let n=0;n<s;n++,t+=1)r[n]=e.getUint8(t);return String.fromCharCode.apply(null,r)},H.UTF16=function(e,t,n){const r=[],s=n/2;for(let n=0;n<s;n++,t+=2)r[n]=e.getUint16(t);return String.fromCharCode.apply(null,r)},z.UTF16=function(e){const t=[];for(let n=0;n<e.length;n+=1){const r=e.charCodeAt(n);t[t.length]=r>>8&255,t[t.length]=255&r}return t},V.UTF16=function(e){return 2*e.length};var _={"x-mac-croatian":"ÄÅÇÉÑÖÜáàâäãåçéèêëíìîïñóòôöõúùûü†°¢£§•¶ß®Š™´¨≠ŽØ∞±≤≥∆µ∂∑∏š∫ªºΩžø¿¡¬√ƒ≈Ć«Č… ÀÃÕŒœĐ—“”‘’÷◊©⁄€‹›Æ»–·‚„‰ÂćÁčÈÍÎÏÌÓÔđÒÚÛÙıˆ˜¯πË˚¸Êæˇ","x-mac-cyrillic":"АБВГДЕЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯ†°Ґ£§•¶І®©™Ђђ≠Ѓѓ∞±≤≥іµґЈЄєЇїЉљЊњјЅ¬√ƒ≈∆«»… ЋћЌќѕ–—“”‘’÷„ЎўЏџ№Ёёяабвгдежзийклмнопрстуфхцчшщъыьэю","x-mac-gaelic":"ÄÅÇÉÑÖÜáàâäãåçéèêëíìîïñóòôöõúùûü†°¢£§•¶ß®©™´¨≠ÆØḂ±≤≥ḃĊċḊḋḞḟĠġṀæøṁṖṗɼƒſṠ«»… ÀÃÕŒœ–—“”‘’ṡẛÿŸṪ€‹›Ŷŷṫ·Ỳỳ⁊ÂÊÁËÈÍÎÏÌÓÔ♣ÒÚÛÙıÝýŴŵẄẅẀẁẂẃ","x-mac-greek":"Ä¹²É³ÖÜ΅àâä΄¨çéèêë£™îï•½‰ôö¦€ùûü†ΓΔΘΛΞΠß®©ΣΪ§≠°·Α±≤≥¥ΒΕΖΗΙΚΜΦΫΨΩάΝ¬ΟΡ≈Τ«»… ΥΧΆΈœ–―“”‘’÷ΉΊΌΎέήίόΏύαβψδεφγηιξκλμνοπώρστθωςχυζϊϋΐΰ­","x-mac-icelandic":"ÄÅÇÉÑÖÜáàâäãåçéèêëíìîïñóòôöõúùûüÝ°¢£§•¶ß®©™´¨≠ÆØ∞±≤≥¥µ∂∑∏π∫ªºΩæø¿¡¬√ƒ≈∆«»… ÀÃÕŒœ–—“”‘’÷◊ÿŸ⁄€ÐðÞþý·‚„‰ÂÊÁËÈÍÎÏÌÓÔÒÚÛÙıˆ˜¯˘˙˚¸˝˛ˇ","x-mac-inuit":"ᐃᐄᐅᐆᐊᐋᐱᐲᐳᐴᐸᐹᑉᑎᑏᑐᑑᑕᑖᑦᑭᑮᑯᑰᑲᑳᒃᒋᒌᒍᒎᒐᒑ°ᒡᒥᒦ•¶ᒧ®©™ᒨᒪᒫᒻᓂᓃᓄᓅᓇᓈᓐᓯᓰᓱᓲᓴᓵᔅᓕᓖᓗᓘᓚᓛᓪᔨᔩᔪᔫᔭ… ᔮᔾᕕᕖᕗ–—“”‘’ᕘᕙᕚᕝᕆᕇᕈᕉᕋᕌᕐᕿᖀᖁᖂᖃᖄᖅᖏᖐᖑᖒᖓᖔᖕᙱᙲᙳᙴᙵᙶᖖᖠᖡᖢᖣᖤᖥᖦᕼŁł","x-mac-ce":"ÄĀāÉĄÖÜáąČäčĆćéŹźĎíďĒēĖóėôöõúĚěü†°Ę£§•¶ß®©™ę¨≠ģĮįĪ≤≥īĶ∂∑łĻļĽľĹĺŅņŃ¬√ńŇ∆«»… ňŐÕőŌ–—“”‘’÷◊ōŔŕŘ‹›řŖŗŠ‚„šŚśÁŤťÍŽžŪÓÔūŮÚůŰűŲųÝýķŻŁżĢˇ",macintosh:"ÄÅÇÉÑÖÜáàâäãåçéèêëíìîïñóòôöõúùûü†°¢£§•¶ß®©™´¨≠ÆØ∞±≤≥¥µ∂∑∏π∫ªºΩæø¿¡¬√ƒ≈∆«»… ÀÃÕŒœ–—“”‘’÷◊ÿŸ⁄€‹›ﬁﬂ‡·‚„‰ÂÊÁËÈÍÎÏÌÓÔÒÚÛÙıˆ˜¯˘˙˚¸˝˛ˇ","x-mac-romanian":"ÄÅÇÉÑÖÜáàâäãåçéèêëíìîïñóòôöõúùûü†°¢£§•¶ß®©™´¨≠ĂȘ∞±≤≥¥µ∂∑∏π∫ªºΩăș¿¡¬√ƒ≈∆«»… ÀÃÕŒœ–—“”‘’÷◊ÿŸ⁄€‹›Țț‡·‚„‰ÂÊÁËÈÍÎÏÌÓÔÒÚÛÙıˆ˜¯˘˙˚¸˝˛ˇ","x-mac-turkish":"ÄÅÇÉÑÖÜáàâäãåçéèêëíìîïñóòôöõúùûü†°¢£§•¶ß®©™´¨≠ÆØ∞±≤≥¥µ∂∑∏π∫ªºΩæø¿¡¬√ƒ≈∆«»… ÀÃÕŒœ–—“”‘’÷◊ÿŸĞğİıŞş‡·‚„‰ÂÊÁËÈÍÎÏÌÓÔÒÚÛÙˆ˜¯˘˙˚¸˝˛ˇ"};H.MACSTRING=function(e,t,n,r){const s=_[r];if(void 0===s)return;let o="";for(let r=0;r<n;r++){const n=e.getUint8(t+r);o+=n<=127?String.fromCharCode(n):s[127&n]}return o};var j,W="function"==typeof WeakMap&&new WeakMap;function $(e){return e>=-128&&e<=127}function Y(e,t,n){let r=0;const s=e.length;for(;t<s&&r<64&&0===e[t];)++t,++r;return n.push(128|r-1),t}function X(e,t,n){let r=0;const s=e.length;let o=t;for(;o<s&&r<64;){const t=e[o];if(!$(t))break;if(0===t&&o+1<s&&0===e[o+1])break;++o,++r}n.push(r-1);for(let r=t;r<o;++r)n.push(e[r]+256&255);return o}function Z(e,t,n){let r=0;const s=e.length;let o=t;for(;o<s&&r<64;){const t=e[o];if(0===t)break;if($(t)&&o+1<s&&$(e[o+1]))break;++o,++r}n.push(64|r-1);for(let r=t;r<o;++r){const t=e[r];n.push(t+65536>>8&255,t+256&255)}return o}z.MACSTRING=function(e,t){const n=function(e){if(!j){j={};for(let e in _)j[e]=new String(e)}const t=j[e];if(void 0===t)return;if(W){const e=W.get(t);if(void 0!==e)return e}const n=_[e];if(void 0===n)return;const r={};for(let e=0;e<n.length;e++)r[n.charCodeAt(e)]=e+128;return W&&W.set(t,r),r}(t);if(void 0===n)return;const r=[];for(let t=0;t<e.length;t++){let s=e.charCodeAt(t);if(s>=128&&(s=n[s],void 0===s))return;r[t]=s}return r},V.MACSTRING=function(e,t){const n=z.MACSTRING(e,t);return void 0!==n?n.length:0},z.VARDELTAS=function(e){let t=0;const n=[];for(;t<e.length;){const r=e[t];t=0===r?Y(e,t,n):r>=-128&&r<=127?X(e,t,n):Z(e,t,n)}return n},z.INDEX=function(e){let t=1;const n=[t],r=[];for(let s=0;s<e.length;s+=1){const o=z.OBJECT(e[s]);Array.prototype.push.apply(r,o),t+=o.length,n.push(t)}if(0===r.length)return[0,0];const s=[],o=1+Math.floor(Math.log(t)/Math.log(2))/8|0,a=[void 0,z.BYTE,z.USHORT,z.UINT24,z.ULONG][o];for(let e=0;e<n.length;e+=1){const t=a(n[e]);Array.prototype.push.apply(s,t)}return Array.prototype.concat(z.Card16(e.length),z.OffSize(o),s,r)},V.INDEX=function(e){return z.INDEX(e).length},z.DICT=function(e){let t=[];const n=Object.keys(e),r=n.length;for(let s=0;s<r;s+=1){const r=parseInt(n[s],0),o=e[r],a=z.OPERAND(o.value,o.type),i=z.OPERATOR(r);for(let e=0;e<a.length;e++)t.push(a[e]);for(let e=0;e<i.length;e++)t.push(i[e])}return t},V.DICT=function(e){return z.DICT(e).length},z.OPERATOR=function(e){return e<1200?[e]:[12,e-1200]},z.OPERAND=function(e,t){let n=[];if(Array.isArray(t))for(let r=0;r<t.length;r+=1){G.argument(e.length===t.length,"Not enough arguments given for type"+t);const s=z.OPERAND(e[r],t[r]);for(let e=0;e<s.length;e++)n.push(s[e])}else if("SID"===t){const t=z.NUMBER(e);for(let e=0;e<t.length;e++)n.push(t[e])}else if("offset"===t){const t=z.NUMBER32(e);for(let e=0;e<t.length;e++)n.push(t[e])}else if("number"===t){const t=z.NUMBER(e);for(let e=0;e<t.length;e++)n.push(t[e])}else{if("real"!==t)throw new Error("Unknown operand type "+t);{const t=z.REAL(e);for(let e=0;e<t.length;e++)n.push(t[e])}}return n},z.OP=z.BYTE,V.OP=V.BYTE;var Q="function"==typeof WeakMap&&new WeakMap;function K(e,t,n){if(t&&t.length)for(let e=0;e<t.length;e+=1){const n=t[e];this[n.name]=n.value}if(this.tableName=e,this.fields=t,n){const e=Object.keys(n);for(let t=0;t<e.length;t+=1){const r=e[t],s=n[r];void 0!==this[r]&&(this[r]=s)}}}function J(e,t,n){void 0===n&&(n=t.length);const r=new Array(t.length+1);r[0]={name:e+"Count",type:"USHORT",value:n};for(let n=0;n<t.length;n++)r[n+1]={name:e+n,type:"USHORT",value:t[n]};return r}function ee(e,t,n){const r=t.length,s=new Array(r+1);s[0]={name:e+"Count",type:"USHORT",value:r};for(let o=0;o<r;o++)s[o+1]={name:e+o,type:"TABLE",value:n(t[o],o)};return s}function te(e,t,n){const r=t.length;let s=[];s[0]={name:e+"Count",type:"USHORT",value:r};for(let e=0;e<r;e++)s=s.concat(n(t[e],e));return s}function ne(e){1===e.format?K.call(this,"coverageTable",[{name:"coverageFormat",type:"USHORT",value:1}].concat(J("glyph",e.glyphs))):2===e.format?K.call(this,"coverageTable",[{name:"coverageFormat",type:"USHORT",value:2}].concat(te("rangeRecord",e.ranges,(function(e,t){return[{name:"startGlyphID"+t,type:"USHORT",value:e.start},{name:"endGlyphID"+t,type:"USHORT",value:e.end},{name:"startCoverageIndex"+t,type:"USHORT",value:e.index}]})))):G.assert(!1,"Coverage format must be 1 or 2.")}function re(e){K.call(this,"scriptListTable",te("scriptRecord",e,(function(e,t){const n=e.script;let r=n.defaultLangSys;return G.assert(!!r,"Unable to write GSUB: script "+e.tag+" has no default language system."),[{name:"scriptTag"+t,type:"TAG",value:e.tag},{name:"script"+t,type:"TABLE",value:new K("scriptTable",[{name:"defaultLangSys",type:"TABLE",value:new K("defaultLangSys",[{name:"lookupOrder",type:"USHORT",value:0},{name:"reqFeatureIndex",type:"USHORT",value:r.reqFeatureIndex}].concat(J("featureIndex",r.featureIndexes)))}].concat(te("langSys",n.langSysRecords,(function(e,t){const n=e.langSys;return[{name:"langSysTag"+t,type:"TAG",value:e.tag},{name:"langSys"+t,type:"TABLE",value:new K("langSys",[{name:"lookupOrder",type:"USHORT",value:0},{name:"reqFeatureIndex",type:"USHORT",value:n.reqFeatureIndex}].concat(J("featureIndex",n.featureIndexes)))}]}))))}]})))}function se(e){K.call(this,"featureListTable",te("featureRecord",e,(function(e,t){const n=e.feature;return[{name:"featureTag"+t,type:"TAG",value:e.tag},{name:"feature"+t,type:"TABLE",value:new K("featureTable",[{name:"featureParams",type:"USHORT",value:n.featureParams}].concat(J("lookupListIndex",n.lookupListIndexes)))}]})))}function oe(e,t){K.call(this,"lookupListTable",ee("lookup",e,(function(e){let n=t[e.lookupType];return G.assert(!!n,"Unable to write GSUB lookup type "+e.lookupType+" tables."),new K("lookupTable",[{name:"lookupType",type:"USHORT",value:e.lookupType},{name:"lookupFlag",type:"USHORT",value:e.lookupFlag}].concat(ee("subtable",e.subtables,n)))})))}function ae(e){1===e.format?K.call(this,"classDefTable",[{name:"classFormat",type:"USHORT",value:1},{name:"startGlyphID",type:"USHORT",value:e.startGlyph}].concat(J("glyph",e.classes))):2===e.format?K.call(this,"classDefTable",[{name:"classFormat",type:"USHORT",value:2}].concat(te("rangeRecord",e.ranges,(function(e,t){return[{name:"startGlyphID"+t,type:"USHORT",value:e.start},{name:"endGlyphID"+t,type:"USHORT",value:e.end},{name:"class"+t,type:"USHORT",value:e.classId}]})))):G.assert(!1,"Class format must be 1 or 2.")}z.CHARSTRING=function(e){if(Q){const t=Q.get(e);if(void 0!==t)return t}let t=[];const n=e.length;for(let r=0;r<n;r+=1){const n=e[r],s=z[n.type](n.value);for(let e=0;e<s.length;e++)t.push(s[e])}return Q&&Q.set(e,t),t},V.CHARSTRING=function(e){return z.CHARSTRING(e).length},z.OBJECT=function(e){const t=z[e.type];return G.argument(void 0!==t,"No encoding function for type "+e.type),t(e.value)},V.OBJECT=function(e){const t=V[e.type];return G.argument(void 0!==t,"No sizeOf function for type "+e.type),t(e.value)},z.TABLE=function(e){let t=[];const n=(e.fields||[]).length,r=[],s=[];for(let o=0;o<n;o+=1){const n=e.fields[o],a=z[n.type];G.argument(void 0!==a,"No encoding function for field type "+n.type+" ("+n.name+")");let i=e[n.name];void 0===i&&(i=n.value);const l=a(i);if("TABLE"===n.type)null!==i.fields&&(s.push(t.length),r.push(l)),t.push(0,0);else for(let e=0;e<l.length;e++)t.push(l[e])}for(let n=0;n<r.length;n+=1){const o=s[n],a=t.length;G.argument(a<65536,"Table "+e.tableName+" too big."),t[o]=a>>8,t[o+1]=255&a;for(let e=0;e<r[n].length;e++)t.push(r[n][e])}return t},V.TABLE=function(e){let t=0;const n=(e.fields||[]).length;for(let r=0;r<n;r+=1){const n=e.fields[r],s=V[n.type];G.argument(void 0!==s,"No sizeOf function for field type "+n.type+" ("+n.name+")");let o=e[n.name];void 0===o&&(o=n.value),t+=s(o),"TABLE"===n.type&&(t+=2)}return t},z.RECORD=z.TABLE,V.RECORD=V.TABLE,z.LITERAL=function(e){return e},V.LITERAL=function(e){return e.length},K.prototype.encode=function(){return z.TABLE(this)},K.prototype.sizeOf=function(){return V.TABLE(this)},ne.prototype=Object.create(K.prototype),ne.prototype.constructor=ne,re.prototype=Object.create(K.prototype),re.prototype.constructor=re,se.prototype=Object.create(K.prototype),se.prototype.constructor=se,oe.prototype=Object.create(K.prototype),oe.prototype.constructor=oe,ae.prototype=Object.create(K.prototype),ae.prototype.constructor=ae;var ie={Table:K,Record:K,Coverage:ne,ClassDef:ae,ScriptList:re,FeatureList:se,LookupList:oe,ushortList:J,tableList:ee,recordList:te};function le(e,t){return e.getUint8(t)}function ue(e,t){return e.getUint16(t,!1)}function ce(e,t){return(e.getUint16(t)<<8)+e.getUint8(t+2)}function pe(e,t){return e.getUint32(t,!1)}function he(e,t){return e.getInt16(t,!1)+e.getUint16(t+2,!1)/65535}var fe={byte:1,uShort:2,f2dot14:2,short:2,uInt24:3,uLong:4,fixed:4,longDateTime:8,tag:4},de=32768,ge=32767,me=32768,ye=4095,ve=32768,be=16384,Se=8192,xe=4095,Te=128,ke=127,Oe=128,Ue=64,Re=63,we=15,Ie=48;function Ce(e,t){this.data=e,this.offset=t,this.relativeOffset=0}Ce.prototype.parseByte=function(){const e=this.data.getUint8(this.offset+this.relativeOffset);return this.relativeOffset+=1,e},Ce.prototype.parseChar=function(){const e=this.data.getInt8(this.offset+this.relativeOffset);return this.relativeOffset+=1,e},Ce.prototype.parseCard8=Ce.prototype.parseByte,Ce.prototype.parseUShort=function(){const e=this.data.getUint16(this.offset+this.relativeOffset);return this.relativeOffset+=2,e},Ce.prototype.parseCard16=Ce.prototype.parseUShort,Ce.prototype.parseSID=Ce.prototype.parseUShort,Ce.prototype.parseOffset16=Ce.prototype.parseUShort,Ce.prototype.parseShort=function(){const e=this.data.getInt16(this.offset+this.relativeOffset);return this.relativeOffset+=2,e},Ce.prototype.parseF2Dot14=function(){const e=this.data.getInt16(this.offset+this.relativeOffset)/16384;return this.relativeOffset+=2,e},Ce.prototype.parseUInt24=function(){const e=ce(this.data,this.offset+this.relativeOffset);return this.relativeOffset+=3,e},Ce.prototype.parseULong=function(){const e=pe(this.data,this.offset+this.relativeOffset);return this.relativeOffset+=4,e},Ce.prototype.parseLong=function(){const e=(t=this.data,n=this.offset+this.relativeOffset,t.getInt32(n,!1));var t,n;return this.relativeOffset+=4,e},Ce.prototype.parseOffset32=Ce.prototype.parseULong,Ce.prototype.parseFixed=function(){const e=he(this.data,this.offset+this.relativeOffset);return this.relativeOffset+=4,e},Ce.prototype.parseString=function(e){const t=this.data,n=this.offset+this.relativeOffset;let r="";this.relativeOffset+=e;for(let s=0;s<e;s++)r+=String.fromCharCode(t.getUint8(n+s));return r},Ce.prototype.parseTag=function(){return this.parseString(4)},Ce.prototype.parseLongDateTime=function(){let e=pe(this.data,this.offset+this.relativeOffset+4);return e-=2082844800,this.relativeOffset+=8,e},Ce.prototype.parseVersion=function(e){const t=ue(this.data,this.offset+this.relativeOffset),n=ue(this.data,this.offset+this.relativeOffset+2);return this.relativeOffset+=4,void 0===e&&(e=4096),t+n/e/10},Ce.prototype.skip=function(e,t){void 0===t&&(t=1),this.relativeOffset+=fe[e]*t},Ce.prototype.parseULongList=function(e){void 0===e&&(e=this.parseULong());const t=new Array(e),n=this.data;let r=this.offset+this.relativeOffset;for(let s=0;s<e;s++)t[s]=n.getUint32(r),r+=4;return this.relativeOffset+=4*e,t},Ce.prototype.parseOffset16List=Ce.prototype.parseUShortList=function(e){void 0===e&&(e=this.parseUShort());const t=new Array(e),n=this.data;let r=this.offset+this.relativeOffset;for(let s=0;s<e;s++)t[s]=n.getUint16(r),r+=2;return this.relativeOffset+=2*e,t},Ce.prototype.parseShortList=function(e){const t=new Array(e),n=this.data;let r=this.offset+this.relativeOffset;for(let s=0;s<e;s++)t[s]=n.getInt16(r),r+=2;return this.relativeOffset+=2*e,t},Ce.prototype.parseByteList=function(e){const t=new Array(e),n=this.data;let r=this.offset+this.relativeOffset;for(let s=0;s<e;s++)t[s]=n.getUint8(r++);return this.relativeOffset+=e,t},Ce.prototype.parseList=function(e,t){t||(t=e,e=this.parseUShort());const n=new Array(e);for(let r=0;r<e;r++)n[r]=t.call(this);return n},Ce.prototype.parseList32=function(e,t){t||(t=e,e=this.parseULong());const n=new Array(e);for(let r=0;r<e;r++)n[r]=t.call(this);return n},Ce.prototype.parseRecordList=function(e,t){t||(t=e,e=this.parseUShort());const n=new Array(e),r=Object.keys(t);for(let s=0;s<e;s++){const e={};for(let n=0;n<r.length;n++){const s=r[n],o=t[s];e[s]=o.call(this)}n[s]=e}return n},Ce.prototype.parseRecordList32=function(e,t){t||(t=e,e=this.parseULong());const n=new Array(e),r=Object.keys(t);for(let s=0;s<e;s++){const e={};for(let n=0;n<r.length;n++){const s=r[n],o=t[s];e[s]=o.call(this)}n[s]=e}return n},Ce.prototype.parseTupleRecords=function(e,t){let n=[];for(let r=0;r<e;r++){let e=[];for(let n=0;n<t;n++)e.push(this.parseF2Dot14());n.push(e)}return n},Ce.prototype.parseStruct=function(e){if("function"==typeof e)return e.call(this);{const t=Object.keys(e),n={};for(let r=0;r<t.length;r++){const s=t[r],o=e[s];n[s]=o.call(this)}return n}},Ce.prototype.parseValueRecord=function(e){if(void 0===e&&(e=this.parseUShort()),0===e)return;const t={};return 1&e&&(t.xPlacement=this.parseShort()),2&e&&(t.yPlacement=this.parseShort()),4&e&&(t.xAdvance=this.parseShort()),8&e&&(t.yAdvance=this.parseShort()),16&e&&(t.xPlaDevice=void 0,this.parseShort()),32&e&&(t.yPlaDevice=void 0,this.parseShort()),64&e&&(t.xAdvDevice=void 0,this.parseShort()),128&e&&(t.yAdvDevice=void 0,this.parseShort()),t},Ce.prototype.parseValueRecordList=function(){const e=this.parseUShort(),t=this.parseUShort(),n=new Array(t);for(let r=0;r<t;r++)n[r]=this.parseValueRecord(e);return n},Ce.prototype.parsePointer=function(e){const t=this.parseOffset16();if(t>0)return new Ce(this.data,this.offset+t).parseStruct(e)},Ce.prototype.parsePointer32=function(e){const t=this.parseOffset32();if(t>0)return new Ce(this.data,this.offset+t).parseStruct(e)},Ce.prototype.parseListOfLists=function(e){const t=this.parseOffset16List(),n=t.length,r=this.relativeOffset,s=new Array(n);for(let r=0;r<n;r++){const n=t[r];if(0!==n)if(this.relativeOffset=n,e){const t=this.parseOffset16List(),o=new Array(t.length);for(let r=0;r<t.length;r++)this.relativeOffset=n+t[r],o[r]=e.call(this);s[r]=o}else s[r]=this.parseUShortList();else s[r]=void 0}return this.relativeOffset=r,s},Ce.prototype.parseCoverage=function(){const e=this.offset+this.relativeOffset,t=this.parseUShort(),n=this.parseUShort();if(1===t)return{format:1,glyphs:this.parseUShortList(n)};if(2===t){const e=new Array(n);for(let t=0;t<n;t++)e[t]={start:this.parseUShort(),end:this.parseUShort(),index:this.parseUShort()};return{format:2,ranges:e}}throw new Error("0x"+e.toString(16)+": Coverage format must be 1 or 2.")},Ce.prototype.parseClassDef=function(){const e=this.offset+this.relativeOffset,t=this.parseUShort();return 1===t?{format:1,startGlyph:this.parseUShort(),classes:this.parseUShortList()}:2===t?{format:2,ranges:this.parseRecordList({start:Ce.uShort,end:Ce.uShort,classId:Ce.uShort})}:(console.warn(`0x${e.toString(16)}: This font file uses an invalid ClassDef format of ${t}. It might be corrupted and should be reacquired if it doesn't display as intended.`),{format:t})},Ce.list=function(e,t){return function(){return this.parseList(e,t)}},Ce.list32=function(e,t){return function(){return this.parseList32(e,t)}},Ce.recordList=function(e,t){return function(){return this.parseRecordList(e,t)}},Ce.recordList32=function(e,t){return function(){return this.parseRecordList32(e,t)}},Ce.pointer=function(e){return function(){return this.parsePointer(e)}},Ce.pointer32=function(e){return function(){return this.parsePointer32(e)}},Ce.tag=Ce.prototype.parseTag,Ce.byte=Ce.prototype.parseByte,Ce.uShort=Ce.offset16=Ce.prototype.parseUShort,Ce.uShortList=Ce.prototype.parseUShortList,Ce.uInt24=Ce.prototype.parseUInt24,Ce.uLong=Ce.offset32=Ce.prototype.parseULong,Ce.uLongList=Ce.prototype.parseULongList,Ce.fixed=Ce.prototype.parseFixed,Ce.f2Dot14=Ce.prototype.parseF2Dot14,Ce.struct=Ce.prototype.parseStruct,Ce.coverage=Ce.prototype.parseCoverage,Ce.classDef=Ce.prototype.parseClassDef;var Le={reserved:Ce.uShort,reqFeatureIndex:Ce.uShort,featureIndexes:Ce.uShortList};Ce.prototype.parseScriptList=function(){return this.parsePointer(Ce.recordList({tag:Ce.tag,script:Ce.pointer({defaultLangSys:Ce.pointer(Le),langSysRecords:Ce.recordList({tag:Ce.tag,langSys:Ce.pointer(Le)})})}))||[]},Ce.prototype.parseFeatureList=function(){return this.parsePointer(Ce.recordList({tag:Ce.tag,feature:Ce.pointer({featureParams:Ce.offset16,lookupListIndexes:Ce.uShortList})}))||[]},Ce.prototype.parseLookupList=function(e){return this.parsePointer(Ce.list(Ce.pointer((function(){const t=this.parseUShort();G.argument(1<=t&&t<=9,"GPOS/GSUB lookup type "+t+" unknown.");const n=this.parseUShort(),r=16&n;return{lookupType:t,lookupFlag:n,subtables:this.parseList(Ce.pointer(e[t])),markFilteringSet:r?this.parseUShort():void 0}}))))||[]},Ce.prototype.parseFeatureVariationsList=function(){return this.parsePointer32((function(){const e=this.parseUShort(),t=this.parseUShort();G.argument(1===e&&t<1,"GPOS/GSUB feature variations table unknown.");return this.parseRecordList32({conditionSetOffset:Ce.offset32,featureTableSubstitutionOffset:Ce.offset32})}))||[]},Ce.prototype.parseVariationStore=function(){const e=this.relativeOffset,t=this.parseUShort(),n={itemVariationStore:this.parseItemVariationStore()};return this.relativeOffset=e+t+2,n},Ce.prototype.parseItemVariationStore=function(){const e=this.relativeOffset,t={format:this.parseUShort(),variationRegions:[],itemVariationSubtables:[]},n=this.parseOffset32(),r=this.parseUShort(),s=this.parseULongList(r);this.relativeOffset=e+n,t.variationRegions=this.parseVariationRegionList();for(let n=0;n<r;n++){const r=s[n];this.relativeOffset=e+r,t.itemVariationSubtables.push(this.parseItemVariationSubtable())}return t},Ce.prototype.parseVariationRegionList=function(){const e=this.parseUShort(),t=this.parseUShort();return this.parseRecordList(t,{regionAxes:Ce.recordList(e,{startCoord:Ce.f2Dot14,peakCoord:Ce.f2Dot14,endCoord:Ce.f2Dot14})})},Ce.prototype.parseItemVariationSubtable=function(){const e=this.parseUShort(),t=this.parseUShort(),n=this.parseUShortList(),r=n.length;return{regionIndexes:n,deltaSets:e&&r?this.parseDeltaSets(e,t,r):[]}},Ce.prototype.parseDeltaSetIndexMap=function(){const e=this.parseByte(),t=this.parseByte(),n=[];let r=0;switch(e){case 0:r=this.parseUShort();break;case 1:r=this.parseULong();break;default:console.error(`unsupported DeltaSetIndexMap format ${e}`)}if(!r)return{format:e,entryFormat:t};const s=1+(t&we),o=1+((t&Ie)>>4);for(let e=0;e<r;e++){let e;if(1===o)e=this.parseByte();else if(2===o)e=this.parseUShort();else if(3===o)e=this.parseUInt24();else{if(4!==o)throw new Error(`Invalid entry size of ${o}`);e=this.getULong()}const t=e>>s,r=e&(1<<s)-1;n.push({outerIndex:t,innerIndex:r})}return{format:e,entryFormat:t,map:n}},Ce.prototype.parseDeltaSets=function(e,t,n){const r=Array.from({length:e},(()=>[])),s=t&de,o=t&ge;if(o>n)throw Error("wordCount must be less than or equal to regionIndexCount");const a=(s?this.parseLong:this.parseShort).bind(this),i=(s?this.parseShort:this.parseChar).bind(this);for(let t=0;t<e;t++)for(let e=0;e<n;e++)e<o?r[t].push(a()):r[t].push(i());return r},Ce.prototype.parseTupleVariationStoreList=function(e,t,n){const r=this.parseUShort(),s=1&this.parseUShort(),o=this.parseOffset32(),a=(s?this.parseULong:this.parseUShort).bind(this),i={};let l,u=a();s||(u*=2);for(let c=0;c<r;c++){l=a(),s||(l*=2);const r=l-u;i[c]=r?this.parseTupleVariationStore(o+u,e,t,n,c):void 0,u=l}return i},Ce.prototype.parseTupleVariationStore=function(e,t,n,r,s){const o=this.relativeOffset;this.relativeOffset=e,"cvar"===n&&(this.relativeOffset+=4);const a=this.parseUShort(),i=!!(a&me),l=a&ye;let u=this.parseOffset16();const c=[];let p=[];for(let e=0;e<l;e++){const e=this.parseTupleVariationHeader(t,n);c.push(e)}this.relativeOffset!==e+u&&(console.warn(`Unexpected offset after parsing tuple variation headers! Expected ${e+u}, actually ${this.relativeOffset}`),this.relativeOffset=e+u),i&&(p=this.parsePackedPointNumbers());let h=this.relativeOffset;for(let e=0;e<l;e++){const t=c[e];t.privatePoints=[],this.relativeOffset=h,"cvar"!==n||t.peakTuple||console.warn("An embedded peak tuple is required in TupleVariationHeaders for the cvar table."),t.flags.privatePointNumbers&&(t.privatePoints=this.parsePackedPointNumbers()),delete t.flags;const o=this.offset,a=this.relativeOffset,i=e=>{let i,l;const u=()=>{let e=0;if("gvar"===n){if(e=t.privatePoints.length||p.length,!e){const t=r.get(s);t.path,e=t.points.length,e+=4}}else"cvar"===n&&(e=r.length);this.offset=o,this.relativeOffset=a,i=this.parsePackedDeltas(e),"gvar"===n&&(l=this.parsePackedDeltas(e))};return{configurable:!0,get:function(){return void 0===i&&u(),"deltasY"===e?l:i},set:function(t){void 0===i&&u(),"deltasY"===e?l=t:i=t}}};Object.defineProperty(t,"deltas",i.call(this,"deltas")),"gvar"===n&&Object.defineProperty(t,"deltasY",i.call(this,"deltasY")),h+=t.variationDataSize,delete t.variationDataSize}this.relativeOffset=o;const f={headers:c};return f.sharedPoints=p,f},Ce.prototype.parseTupleVariationHeader=function(e,t){const n=this.parseUShort(),r=this.parseUShort(),s=!!(r&ve),o=!!(r&be),a=!!(r&Se),i=s?void 0:r&xe,l={variationDataSize:n,peakTuple:s?this.parseTupleRecords(1,e)[0]:void 0,intermediateStartTuple:o?this.parseTupleRecords(1,e)[0]:void 0,intermediateEndTuple:o?this.parseTupleRecords(1,e)[0]:void 0,flags:{embeddedPeakTuple:s,intermediateRegion:o,privatePointNumbers:a}};return"gvar"===t&&(l.sharedTupleRecordsIndex=i),l},Ce.prototype.parsePackedPointNumbers=function(){const e=this.parseByte(),t=[];let n=e;if(e>=128){const t=this.parseByte();n=(e&ke)<<8|t}let r=0;for(;t.length<n;){const e=this.parseByte(),s=!!(e&Te);let o=1+(e&ke);for(let e=0;e<o&&t.length<n;e++){let e;e=s?this.parseUShort():this.parseByte(),r+=e,t.push(r)}}return t},Ce.prototype.parsePackedDeltas=function(e){const t=[];for(;t.length<e;){const n=this.parseByte(),r=!!(n&Oe),s=!!(n&Ue),o=1+(n&Re);for(let n=0;n<o&&t.length<e;n++)r?t.push(0):s?t.push(this.parseShort()):t.push(this.parseChar())}return t};var Ee={getByte:le,getCard8:le,getUShort:ue,getCard16:ue,getShort:function(e,t){return e.getInt16(t,!1)},getUInt24:ce,getULong:pe,getFixed:he,getTag:function(e,t){let n="";for(let r=t;r<t+4;r+=1)n+=String.fromCharCode(e.getInt8(r));return n},getOffset:function(e,t,n){let r=0;for(let s=0;s<n;s+=1)r<<=8,r+=e.getUint8(t+s);return r},getBytes:function(e,t,n){const r=[];for(let s=t;s<n;s+=1)r.push(e.getUint8(s));return r},bytesToString:function(e){let t="";for(let n=0;n<e.length;n+=1)t+=String.fromCharCode(e[n]);return t},Parser:Ce},De=["copyright","fontFamily","fontSubfamily","uniqueID","fullName","version","postScriptName","trademark","manufacturer","designer","description","manufacturerURL","designerURL","license","licenseURL","reserved","preferredFamily","preferredSubfamily","compatibleFullName","sampleText","postScriptFindFontName","wwsFamily","wwsSubfamily"],Me={0:"en",1:"fr",2:"de",3:"it",4:"nl",5:"sv",6:"es",7:"da",8:"pt",9:"no",10:"he",11:"ja",12:"ar",13:"fi",14:"el",15:"is",16:"mt",17:"tr",18:"hr",19:"zh-Hant",20:"ur",21:"hi",22:"th",23:"ko",24:"lt",25:"pl",26:"hu",27:"es",28:"lv",29:"se",30:"fo",31:"fa",32:"ru",33:"zh",34:"nl-BE",35:"ga",36:"sq",37:"ro",38:"cz",39:"sk",40:"si",41:"yi",42:"sr",43:"mk",44:"bg",45:"uk",46:"be",47:"uz",48:"kk",49:"az-Cyrl",50:"az-Arab",51:"hy",52:"ka",53:"mo",54:"ky",55:"tg",56:"tk",57:"mn-CN",58:"mn",59:"ps",60:"ks",61:"ku",62:"sd",63:"bo",64:"ne",65:"sa",66:"mr",67:"bn",68:"as",69:"gu",70:"pa",71:"or",72:"ml",73:"kn",74:"ta",75:"te",76:"si",77:"my",78:"km",79:"lo",80:"vi",81:"id",82:"tl",83:"ms",84:"ms-Arab",85:"am",86:"ti",87:"om",88:"so",89:"sw",90:"rw",91:"rn",92:"ny",93:"mg",94:"eo",128:"cy",129:"eu",130:"ca",131:"la",132:"qu",133:"gn",134:"ay",135:"tt",136:"ug",137:"dz",138:"jv",139:"su",140:"gl",141:"af",142:"br",143:"iu",144:"gd",145:"gv",146:"ga",147:"to",148:"el-polyton",149:"kl",150:"az",151:"nn"},Ae={0:0,1:0,2:0,3:0,4:0,5:0,6:0,7:0,8:0,9:0,10:5,11:1,12:4,13:0,14:6,15:0,16:0,17:0,18:0,19:2,20:4,21:9,22:21,23:3,24:29,25:29,26:29,27:29,28:29,29:0,30:0,31:4,32:7,33:25,34:0,35:0,36:0,37:0,38:29,39:29,40:0,41:5,42:7,43:7,44:7,45:7,46:7,47:7,48:7,49:7,50:4,51:24,52:23,53:7,54:7,55:7,56:7,57:27,58:7,59:4,60:4,61:4,62:4,63:26,64:9,65:9,66:9,67:13,68:13,69:11,70:10,71:12,72:17,73:16,74:14,75:15,76:18,77:19,78:20,79:22,80:30,81:0,82:0,83:0,84:4,85:28,86:28,87:28,88:0,89:0,90:0,91:0,92:0,93:0,94:0,128:0,129:0,130:0,131:0,132:0,133:0,134:0,135:7,136:4,137:26,138:0,139:0,140:0,141:0,142:0,143:28,144:0,145:0,146:0,147:0,148:6,149:0,150:0,151:0},Fe={1078:"af",1052:"sq",1156:"gsw",1118:"am",5121:"ar-DZ",15361:"ar-BH",3073:"ar",2049:"ar-IQ",11265:"ar-JO",13313:"ar-KW",12289:"ar-LB",4097:"ar-LY",6145:"ary",8193:"ar-OM",16385:"ar-QA",1025:"ar-SA",10241:"ar-SY",7169:"aeb",14337:"ar-AE",9217:"ar-YE",1067:"hy",1101:"as",2092:"az-Cyrl",1068:"az",1133:"ba",1069:"eu",1059:"be",2117:"bn",1093:"bn-IN",8218:"bs-Cyrl",5146:"bs",1150:"br",1026:"bg",1027:"ca",3076:"zh-HK",5124:"zh-MO",2052:"zh",4100:"zh-SG",1028:"zh-TW",1155:"co",1050:"hr",4122:"hr-BA",1029:"cs",1030:"da",1164:"prs",1125:"dv",2067:"nl-BE",1043:"nl",3081:"en-AU",10249:"en-BZ",4105:"en-CA",9225:"en-029",16393:"en-IN",6153:"en-IE",8201:"en-JM",17417:"en-MY",5129:"en-NZ",13321:"en-PH",18441:"en-SG",7177:"en-ZA",11273:"en-TT",2057:"en-GB",1033:"en",12297:"en-ZW",1061:"et",1080:"fo",1124:"fil",1035:"fi",2060:"fr-BE",3084:"fr-CA",1036:"fr",5132:"fr-LU",6156:"fr-MC",4108:"fr-CH",1122:"fy",1110:"gl",1079:"ka",3079:"de-AT",1031:"de",5127:"de-LI",4103:"de-LU",2055:"de-CH",1032:"el",1135:"kl",1095:"gu",1128:"ha",1037:"he",1081:"hi",1038:"hu",1039:"is",1136:"ig",1057:"id",1117:"iu",2141:"iu-Latn",2108:"ga",1076:"xh",1077:"zu",1040:"it",2064:"it-CH",1041:"ja",1099:"kn",1087:"kk",1107:"km",1158:"quc",1159:"rw",1089:"sw",1111:"kok",1042:"ko",1088:"ky",1108:"lo",1062:"lv",1063:"lt",2094:"dsb",1134:"lb",1071:"mk",2110:"ms-BN",1086:"ms",1100:"ml",1082:"mt",1153:"mi",1146:"arn",1102:"mr",1148:"moh",1104:"mn",2128:"mn-CN",1121:"ne",1044:"nb",2068:"nn",1154:"oc",1096:"or",1123:"ps",1045:"pl",1046:"pt",2070:"pt-PT",1094:"pa",1131:"qu-BO",2155:"qu-EC",3179:"qu",1048:"ro",1047:"rm",1049:"ru",9275:"smn",4155:"smj-NO",5179:"smj",3131:"se-FI",1083:"se",2107:"se-SE",8251:"sms",6203:"sma-NO",7227:"sms",1103:"sa",7194:"sr-Cyrl-BA",3098:"sr",6170:"sr-Latn-BA",2074:"sr-Latn",1132:"nso",1074:"tn",1115:"si",1051:"sk",1060:"sl",11274:"es-AR",16394:"es-BO",13322:"es-CL",9226:"es-CO",5130:"es-CR",7178:"es-DO",12298:"es-EC",17418:"es-SV",4106:"es-GT",18442:"es-HN",2058:"es-MX",19466:"es-NI",6154:"es-PA",15370:"es-PY",10250:"es-PE",20490:"es-PR",3082:"es",1034:"es",21514:"es-US",14346:"es-UY",8202:"es-VE",2077:"sv-FI",1053:"sv",1114:"syr",1064:"tg",2143:"tzm",1097:"ta",1092:"tt",1098:"te",1054:"th",1105:"bo",1055:"tr",1090:"tk",1152:"ug",1058:"uk",1070:"hsb",1056:"ur",2115:"uz-Cyrl",1091:"uz",1066:"vi",1106:"cy",1160:"wo",1157:"sah",1144:"ii",1130:"yo"};function Pe(e,t,n){switch(e){case 0:if(65535===t)return"und";if(n)return n[t];break;case 1:return Me[t];case 3:return Fe[t]}}var Ne="utf-16",Ge={0:"macintosh",1:"x-mac-japanese",2:"x-mac-chinesetrad",3:"x-mac-korean",6:"x-mac-greek",7:"x-mac-cyrillic",9:"x-mac-devanagai",10:"x-mac-gurmukhi",11:"x-mac-gujarati",12:"x-mac-oriya",13:"x-mac-bengali",14:"x-mac-tamil",15:"x-mac-telugu",16:"x-mac-kannada",17:"x-mac-malayalam",18:"x-mac-sinhalese",19:"x-mac-burmese",20:"x-mac-khmer",21:"x-mac-thai",22:"x-mac-lao",23:"x-mac-georgian",24:"x-mac-armenian",25:"x-mac-chinesesimp",26:"x-mac-tibetan",27:"x-mac-mongolian",28:"x-mac-ethiopic",29:"x-mac-ce",30:"x-mac-vietnamese",31:"x-mac-extarabic"},Be={15:"x-mac-icelandic",17:"x-mac-turkish",18:"x-mac-croatian",24:"x-mac-ce",25:"x-mac-ce",26:"x-mac-ce",27:"x-mac-ce",28:"x-mac-ce",30:"x-mac-icelandic",37:"x-mac-romanian",38:"x-mac-ce",39:"x-mac-ce",40:"x-mac-ce",143:"x-mac-inuit",146:"x-mac-gaelic"};function He(e,t,n){switch(e){case 0:return Ne;case 1:return Be[n]||Ge[t];case 3:if(1===t||10===t)return Ne}}var ze={0:"unicode",1:"macintosh",2:"reserved",3:"windows"};function Ve(e){return ze[e]}function qe(e){const t={};for(let n in e)t[e[n]]=parseInt(n);return t}function _e(e,t,n,r,s,o){return new ie.Record("NameRecord",[{name:"platformID",type:"USHORT",value:e},{name:"encodingID",type:"USHORT",value:t},{name:"languageID",type:"USHORT",value:n},{name:"nameID",type:"USHORT",value:r},{name:"length",type:"USHORT",value:s},{name:"offset",type:"USHORT",value:o}])}function je(e,t){let n=function(e,t){const n=e.length,r=t.length-n+1;e:for(let s=0;s<r;s++)for(;s<r;s++){for(let r=0;r<n;r++)if(t[s+r]!==e[r])continue e;return s}return-1}(e,t);if(n<0){n=t.length;let r=0;const s=e.length;for(;r<s;++r)t.push(e[r])}return n}function We(e,t,n=[]){if(t<256&&t in De){if(n.length&&!n.includes(parseInt(t)))return;t=De[t]}for(let n in e)for(let r in e[n])if(r===t||parseInt(r)===t)return e[n][r]}var $e={parse:function(e,t,n){const r={},s=new Ee.Parser(e,t),o=s.parseUShort(),a=s.parseUShort(),i=s.offset+s.parseUShort();for(let t=0;t<a;t++){const t=s.parseUShort(),o=s.parseUShort(),a=s.parseUShort(),l=s.parseUShort(),u=De[l]||l,c=s.parseUShort(),p=s.parseUShort(),h=Pe(t,a,n),f=He(t,o,a),d=Ve(t);if(void 0!==f&&void 0!==h&&void 0!==d){let t;if(t=f===Ne?H.UTF16(e,i+p,c):H.MACSTRING(e,i+p,c,f),t){let e=r[d];void 0===e&&(e=r[d]={});let n=e[u];void 0===n&&(n=e[u]={}),n[h]=t}}}return 1===o&&s.parseUShort(),r},make:function(e,t){const n=qe(ze),r=qe(Me),s=qe(Fe),o=[],a=[];for(let i in e){let l;const u=[],c={},p=qe(De),h=n[i];for(let t in e[i]){let n=p[t];if(void 0===n&&(n=t),l=parseInt(n),isNaN(l))throw new Error('Name table entry "'+t+'" does not exist, see nameTableNames for complete list.');c[l]=e[i][t],u.push(l)}for(let e=0;e<u.length;e++){l=u[e];const n=c[l];for(let e in n){const i=n[e];if(1===h||0===h){let n=r[e],s=Ae[n];const u=He(h,s,n);let c=z.MACSTRING(i,u);if(0===h&&(n=t.indexOf(e),n<0&&(n=t.length,t.push(e)),s=4,c=z.UTF16(i)),void 0!==c){const e=je(c,a);o.push(_e(h,s,n,l,c.length,e))}}if(3===h){const t=s[e];if(void 0!==t){const e=z.UTF16(i),n=je(e,a);o.push(_e(3,1,t,l,e.length,n))}}}}}o.sort((function(e,t){return e.platformID-t.platformID||e.encodingID-t.encodingID||e.languageID-t.languageID||e.nameID-t.nameID}));const i=new ie.Table("name",[{name:"format",type:"USHORT",value:0},{name:"count",type:"USHORT",value:o.length},{name:"stringOffset",type:"USHORT",value:6+12*o.length}]);for(let e=0;e<o.length;e++)i.fields.push({name:"record_"+e,type:"RECORD",value:o[e]});return i.fields.push({name:"strings",type:"LITERAL",value:a}),i},getNameByID:We};function Ye(e,t,n){e.segments.push({end:t,start:t,delta:-(t-n),offset:0,glyphIndex:n})}var Xe={parse:function(e,t){const n={};n.version=Ee.getUShort(e,t),G.argument(0===n.version,"cmap table version should be 0."),n.numTables=Ee.getUShort(e,t+2);let r=null,s=-1,o=-1,a=null,i=null;const l=[0,1,2,3,4,6],u=[0,1,10];for(let c=n.numTables-1;c>=0;c-=1)if(a=Ee.getUShort(e,t+4+8*c),i=Ee.getUShort(e,t+4+8*c+2),3===a&&u.includes(i)||0===a&&l.includes(i)||1===a&&0===i){if(o>0)continue;if(o=Ee.getULong(e,t+4+8*c+4),r)break}else if(0===a&&5===i)if(s=Ee.getULong(e,t+4+8*c+4),r=new Ee.Parser(e,t+s),14!==r.parseUShort())s=-1,r=null;else if(o>0)break;if(-1===o)throw new Error("No valid cmap sub-tables found.");const c=new Ee.Parser(e,t+o);if(n.format=c.parseUShort(),0===n.format)!function(e,t,n,r){e.length=t.parseUShort(),e.language=t.parseUShort()-1;const s=t.parseByteList(e.length),o=Object.assign({},s),a=He(n,r,e.language),i=_[a];for(let e=0;e<i.length;e++)o[i.charCodeAt(e)]=s[128+e];e.glyphIndexMap=o}(n,c,a,i);else if(12===n.format||13===n.format)!function(e,t,n){let r;t.parseUShort(),e.length=t.parseULong(),e.language=t.parseULong(),e.groupCount=r=t.parseULong(),e.glyphIndexMap={};for(let s=0;s<r;s+=1){const r=t.parseULong(),s=t.parseULong();let o=t.parseULong();for(let t=r;t<=s;t+=1)e.glyphIndexMap[t]=o,12===n&&o++}}(n,c,n.format);else{if(4!==n.format)throw new Error("Only format 0 (platformId 1, encodingId 0), 4, 12 and 14 cmap tables are supported (found format "+n.format+", platformId "+a+", encodingId "+i+").");!function(e,t,n,r,s){let o;e.length=t.parseUShort(),e.language=t.parseUShort(),e.segCount=o=t.parseUShort()>>1,t.skip("uShort",3),e.glyphIndexMap={};const a=new Ee.Parser(n,r+s+14),i=new Ee.Parser(n,r+s+16+2*o),l=new Ee.Parser(n,r+s+16+4*o),u=new Ee.Parser(n,r+s+16+6*o);let c=r+s+16+8*o;for(let t=0;t<o-1;t+=1){let t;const r=a.parseUShort(),s=i.parseUShort(),o=l.parseShort(),p=u.parseUShort();for(let a=s;a<=r;a+=1)0!==p?(c=u.offset+u.relativeOffset-2,c+=p,c+=2*(a-s),t=Ee.getUShort(n,c),0!==t&&(t=t+o&65535)):t=a+o&65535,e.glyphIndexMap[a]=t}}(n,c,e,t,o)}return r&&function(e,t){const n={};t.skip("uLong");const r=t.parseULong();for(let e=0;e<r;e+=1){const e=t.parseUInt24(),r={varSelector:e},s=t.parseOffset32(),o=t.parseOffset32(),a=t.relativeOffset;s&&(t.relativeOffset=s,r.defaultUVS=t.parseStruct({ranges:function(){return t.parseRecordList32({startUnicodeValue:t.parseUInt24,additionalCount:t.parseByte})}})),o&&(t.relativeOffset=o,r.nonDefaultUVS=t.parseStruct({uvsMappings:function(){const e={},n=t.parseRecordList32({unicodeValue:t.parseUInt24,glyphID:t.parseUShort});for(let t=0;t<n.length;t+=1)e[n[t].unicodeValue]=n[t];return e}})),n[e]=r,t.relativeOffset=a}e.varSelectorList=n}(n,r),n},make:function(e){let t,n=!0;for(t=e.length-1;t>0;t-=1){if(e.get(t).unicode>65535){console.log("Adding CMAP format 12 (needed!)"),n=!1;break}}let r=[{name:"version",type:"USHORT",value:0},{name:"numTables",type:"USHORT",value:n?1:2},{name:"platformID",type:"USHORT",value:3},{name:"encodingID",type:"USHORT",value:1},{name:"offset",type:"ULONG",value:n?12:20}];n||r.push({name:"cmap12PlatformID",type:"USHORT",value:3},{name:"cmap12EncodingID",type:"USHORT",value:10},{name:"cmap12Offset",type:"ULONG",value:0}),r.push({name:"format",type:"USHORT",value:4},{name:"cmap4Length",type:"USHORT",value:0},{name:"language",type:"USHORT",value:0},{name:"segCountX2",type:"USHORT",value:0},{name:"searchRange",type:"USHORT",value:0},{name:"entrySelector",type:"USHORT",value:0},{name:"rangeShift",type:"USHORT",value:0});const s=new ie.Table("cmap",r);for(s.segments=[],t=0;t<e.length;t+=1){const n=e.get(t);for(let e=0;e<n.unicodes.length;e+=1)Ye(s,n.unicodes[e],t)}s.segments.sort((function(e,t){return e.start-t.start})),function(e){e.segments.push({end:65535,start:65535,delta:1,offset:0})}(s);const o=s.segments.length;let a=0,i=[],l=[],u=[],c=[],p=[],h=[];for(t=0;t<o;t+=1){const e=s.segments[t];e.end<=65535&&e.start<=65535?(i.push({name:"end_"+t,type:"USHORT",value:e.end}),l.push({name:"start_"+t,type:"USHORT",value:e.start}),u.push({name:"idDelta_"+t,type:"SHORT",value:e.delta}),c.push({name:"idRangeOffset_"+t,type:"USHORT",value:e.offset}),void 0!==e.glyphId&&p.push({name:"glyph_"+t,type:"USHORT",value:e.glyphId})):a+=1,n||void 0===e.glyphIndex||(h.push({name:"cmap12Start_"+t,type:"ULONG",value:e.start}),h.push({name:"cmap12End_"+t,type:"ULONG",value:e.end}),h.push({name:"cmap12Glyph_"+t,type:"ULONG",value:e.glyphIndex}))}s.segCountX2=2*(o-a),s.searchRange=2*Math.pow(2,Math.floor(Math.log(o-a)/Math.log(2))),s.entrySelector=Math.log(s.searchRange/2)/Math.log(2),s.rangeShift=s.segCountX2-s.searchRange;for(let e=0;e<i.length;e++)s.fields.push(i[e]);s.fields.push({name:"reservedPad",type:"USHORT",value:0});for(let e=0;e<l.length;e++)s.fields.push(l[e]);for(let e=0;e<u.length;e++)s.fields.push(u[e]);for(let e=0;e<c.length;e++)s.fields.push(c[e]);for(let e=0;e<p.length;e++)s.fields.push(p[e]);if(s.cmap4Length=14+2*i.length+2+2*l.length+2*u.length+2*c.length+2*p.length,!n){const e=16+4*h.length;s.cmap12Offset=20+s.cmap4Length,s.fields.push({name:"cmap12Format",type:"USHORT",value:12},{name:"cmap12Reserved",type:"USHORT",value:0},{name:"cmap12Length",type:"ULONG",value:e},{name:"cmap12Language",type:"ULONG",value:0},{name:"cmap12nGroups",type:"ULONG",value:h.length/3});for(let e=0;e<h.length;e++)s.fields.push(h[e])}return s}},Ze=[".notdef","space","exclam","quotedbl","numbersign","dollar","percent","ampersand","quoteright","parenleft","parenright","asterisk","plus","comma","hyphen","period","slash","zero","one","two","three","four","five","six","seven","eight","nine","colon","semicolon","less","equal","greater","question","at","A","B","C","D","E","F","G","H","I","J","K","L","M","N","O","P","Q","R","S","T","U","V","W","X","Y","Z","bracketleft","backslash","bracketright","asciicircum","underscore","quoteleft","a","b","c","d","e","f","g","h","i","j","k","l","m","n","o","p","q","r","s","t","u","v","w","x","y","z","braceleft","bar","braceright","asciitilde","exclamdown","cent","sterling","fraction","yen","florin","section","currency","quotesingle","quotedblleft","guillemotleft","guilsinglleft","guilsinglright","fi","fl","endash","dagger","daggerdbl","periodcentered","paragraph","bullet","quotesinglbase","quotedblbase","quotedblright","guillemotright","ellipsis","perthousand","questiondown","grave","acute","circumflex","tilde","macron","breve","dotaccent","dieresis","ring","cedilla","hungarumlaut","ogonek","caron","emdash","AE","ordfeminine","Lslash","Oslash","OE","ordmasculine","ae","dotlessi","lslash","oslash","oe","germandbls","onesuperior","logicalnot","mu","trademark","Eth","onehalf","plusminus","Thorn","onequarter","divide","brokenbar","degree","thorn","threequarters","twosuperior","registered","minus","eth","multiply","threesuperior","copyright","Aacute","Acircumflex","Adieresis","Agrave","Aring","Atilde","Ccedilla","Eacute","Ecircumflex","Edieresis","Egrave","Iacute","Icircumflex","Idieresis","Igrave","Ntilde","Oacute","Ocircumflex","Odieresis","Ograve","Otilde","Scaron","Uacute","Ucircumflex","Udieresis","Ugrave","Yacute","Ydieresis","Zcaron","aacute","acircumflex","adieresis","agrave","aring","atilde","ccedilla","eacute","ecircumflex","edieresis","egrave","iacute","icircumflex","idieresis","igrave","ntilde","oacute","ocircumflex","odieresis","ograve","otilde","scaron","uacute","ucircumflex","udieresis","ugrave","yacute","ydieresis","zcaron","exclamsmall","Hungarumlautsmall","dollaroldstyle","dollarsuperior","ampersandsmall","Acutesmall","parenleftsuperior","parenrightsuperior","266 ff","onedotenleader","zerooldstyle","oneoldstyle","twooldstyle","threeoldstyle","fouroldstyle","fiveoldstyle","sixoldstyle","sevenoldstyle","eightoldstyle","nineoldstyle","commasuperior","threequartersemdash","periodsuperior","questionsmall","asuperior","bsuperior","centsuperior","dsuperior","esuperior","isuperior","lsuperior","msuperior","nsuperior","osuperior","rsuperior","ssuperior","tsuperior","ff","ffi","ffl","parenleftinferior","parenrightinferior","Circumflexsmall","hyphensuperior","Gravesmall","Asmall","Bsmall","Csmall","Dsmall","Esmall","Fsmall","Gsmall","Hsmall","Ismall","Jsmall","Ksmall","Lsmall","Msmall","Nsmall","Osmall","Psmall","Qsmall","Rsmall","Ssmall","Tsmall","Usmall","Vsmall","Wsmall","Xsmall","Ysmall","Zsmall","colonmonetary","onefitted","rupiah","Tildesmall","exclamdownsmall","centoldstyle","Lslashsmall","Scaronsmall","Zcaronsmall","Dieresissmall","Brevesmall","Caronsmall","Dotaccentsmall","Macronsmall","figuredash","hypheninferior","Ogoneksmall","Ringsmall","Cedillasmall","questiondownsmall","oneeighth","threeeighths","fiveeighths","seveneighths","onethird","twothirds","zerosuperior","foursuperior","fivesuperior","sixsuperior","sevensuperior","eightsuperior","ninesuperior","zeroinferior","oneinferior","twoinferior","threeinferior","fourinferior","fiveinferior","sixinferior","seveninferior","eightinferior","nineinferior","centinferior","dollarinferior","periodinferior","commainferior","Agravesmall","Aacutesmall","Acircumflexsmall","Atildesmall","Adieresissmall","Aringsmall","AEsmall","Ccedillasmall","Egravesmall","Eacutesmall","Ecircumflexsmall","Edieresissmall","Igravesmall","Iacutesmall","Icircumflexsmall","Idieresissmall","Ethsmall","Ntildesmall","Ogravesmall","Oacutesmall","Ocircumflexsmall","Otildesmall","Odieresissmall","OEsmall","Oslashsmall","Ugravesmall","Uacutesmall","Ucircumflexsmall","Udieresissmall","Yacutesmall","Thornsmall","Ydieresissmall","001.000","001.001","001.002","001.003","Black","Bold","Book","Light","Medium","Regular","Roman","Semibold"],Qe=[".notdef","space","exclam","quotedbl","numbersign","dollar","percent","ampersand","quoteright","parenleft","parenright","asterisk","plus","comma","hyphen","period","slash","zero","one","two","three","four","five","six","seven","eight","nine","colon","semicolon","less","equal","greater","question","at","A","B","C","D","E","F","G","H","I","J","K","L","M","N","O","P","Q","R","S","T","U","V","W","X","Y","Z","bracketleft","backslash","bracketright","asciicircum","underscore","quoteleft","a","b","c","d","e","f","g","h","i","j","k","l","m","n","o","p","q","r","s","t","u","v","w","x","y","z","braceleft","bar","braceright","asciitilde","exclamdown","cent","sterling","fraction","yen","florin","section","currency","quotesingle","quotedblleft","guillemotleft","guilsinglleft","guilsinglright","fi","fl","endash","dagger","daggerdbl","periodcentered","paragraph","bullet","quotesinglbase","quotedblbase","quotedblright","guillemotright","ellipsis","perthousand","questiondown","grave","acute","circumflex","tilde","macron","breve","dotaccent","dieresis","ring","cedilla","hungarumlaut","ogonek","caron","emdash","AE","ordfeminine","Lslash","Oslash","OE","ordmasculine","ae","dotlessi","lslash","oslash","oe","germandbls","onesuperior","logicalnot","mu","trademark","Eth","onehalf","plusminus","Thorn","onequarter","divide","brokenbar","degree","thorn","threequarters","twosuperior","registered","minus","eth","multiply","threesuperior","copyright","Aacute","Acircumflex","Adieresis","Agrave","Aring","Atilde","Ccedilla","Eacute","Ecircumflex","Edieresis","Egrave","Iacute","Icircumflex","Idieresis","Igrave","Ntilde","Oacute","Ocircumflex","Odieresis","Ograve","Otilde","Scaron","Uacute","Ucircumflex","Udieresis","Ugrave","Yacute","Ydieresis","Zcaron","aacute","acircumflex","adieresis","agrave","aring","atilde","ccedilla","eacute","ecircumflex","edieresis","egrave","iacute","icircumflex","idieresis","igrave","ntilde","oacute","ocircumflex","odieresis","ograve","otilde","scaron","uacute","ucircumflex","udieresis","ugrave","yacute","ydieresis","zcaron"],Ke=[".notdef","space","exclamsmall","Hungarumlautsmall","dollaroldstyle","dollarsuperior","ampersandsmall","Acutesmall","parenleftsuperior","parenrightsuperior","twodotenleader","onedotenleader","comma","hyphen","period","fraction","zerooldstyle","oneoldstyle","twooldstyle","threeoldstyle","fouroldstyle","fiveoldstyle","sixoldstyle","sevenoldstyle","eightoldstyle","nineoldstyle","colon","semicolon","commasuperior","threequartersemdash","periodsuperior","questionsmall","asuperior","bsuperior","centsuperior","dsuperior","esuperior","isuperior","lsuperior","msuperior","nsuperior","osuperior","rsuperior","ssuperior","tsuperior","ff","fi","fl","ffi","ffl","parenleftinferior","parenrightinferior","Circumflexsmall","hyphensuperior","Gravesmall","Asmall","Bsmall","Csmall","Dsmall","Esmall","Fsmall","Gsmall","Hsmall","Ismall","Jsmall","Ksmall","Lsmall","Msmall","Nsmall","Osmall","Psmall","Qsmall","Rsmall","Ssmall","Tsmall","Usmall","Vsmall","Wsmall","Xsmall","Ysmall","Zsmall","colonmonetary","onefitted","rupiah","Tildesmall","exclamdownsmall","centoldstyle","Lslashsmall","Scaronsmall","Zcaronsmall","Dieresissmall","Brevesmall","Caronsmall","Dotaccentsmall","Macronsmall","figuredash","hypheninferior","Ogoneksmall","Ringsmall","Cedillasmall","onequarter","onehalf","threequarters","questiondownsmall","oneeighth","threeeighths","fiveeighths","seveneighths","onethird","twothirds","zerosuperior","onesuperior","twosuperior","threesuperior","foursuperior","fivesuperior","sixsuperior","sevensuperior","eightsuperior","ninesuperior","zeroinferior","oneinferior","twoinferior","threeinferior","fourinferior","fiveinferior","sixinferior","seveninferior","eightinferior","nineinferior","centinferior","dollarinferior","periodinferior","commainferior","Agravesmall","Aacutesmall","Acircumflexsmall","Atildesmall","Adieresissmall","Aringsmall","AEsmall","Ccedillasmall","Egravesmall","Eacutesmall","Ecircumflexsmall","Edieresissmall","Igravesmall","Iacutesmall","Icircumflexsmall","Idieresissmall","Ethsmall","Ntildesmall","Ogravesmall","Oacutesmall","Ocircumflexsmall","Otildesmall","Odieresissmall","OEsmall","Oslashsmall","Ugravesmall","Uacutesmall","Ucircumflexsmall","Udieresissmall","Yacutesmall","Thornsmall","Ydieresissmall"],Je=[".notdef","space","dollaroldstyle","dollarsuperior","parenleftsuperior","parenrightsuperior","twodotenleader","onedotenleader","comma","hyphen","period","fraction","zerooldstyle","oneoldstyle","twooldstyle","threeoldstyle","fouroldstyle","fiveoldstyle","sixoldstyle","sevenoldstyle","eightoldstyle","nineoldstyle","colon","semicolon","commasuperior","threequartersemdash","periodsuperior","asuperior","bsuperior","centsuperior","dsuperior","esuperior","isuperior","lsuperior","msuperior","nsuperior","osuperior","rsuperior","ssuperior","tsuperior","ff","fi","fl","ffi","ffl","parenleftinferior","parenrightinferior","hyphensuperior","colonmonetary","onefitted","rupiah","centoldstyle","figuredash","hypheninferior","onequarter","onehalf","threequarters","oneeighth","threeeighths","fiveeighths","seveneighths","onethird","twothirds","zerosuperior","onesuperior","twosuperior","threesuperior","foursuperior","fivesuperior","sixsuperior","sevensuperior","eightsuperior","ninesuperior","zeroinferior","oneinferior","twoinferior","threeinferior","fourinferior","fiveinferior","sixinferior","seveninferior","eightinferior","nineinferior","centinferior","dollarinferior","periodinferior","commainferior"],et=["","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","space","exclam","quotedbl","numbersign","dollar","percent","ampersand","quoteright","parenleft","parenright","asterisk","plus","comma","hyphen","period","slash","zero","one","two","three","four","five","six","seven","eight","nine","colon","semicolon","less","equal","greater","question","at","A","B","C","D","E","F","G","H","I","J","K","L","M","N","O","P","Q","R","S","T","U","V","W","X","Y","Z","bracketleft","backslash","bracketright","asciicircum","underscore","quoteleft","a","b","c","d","e","f","g","h","i","j","k","l","m","n","o","p","q","r","s","t","u","v","w","x","y","z","braceleft","bar","braceright","asciitilde","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","exclamdown","cent","sterling","fraction","yen","florin","section","currency","quotesingle","quotedblleft","guillemotleft","guilsinglleft","guilsinglright","fi","fl","","endash","dagger","daggerdbl","periodcentered","","paragraph","bullet","quotesinglbase","quotedblbase","quotedblright","guillemotright","ellipsis","perthousand","","questiondown","","grave","acute","circumflex","tilde","macron","breve","dotaccent","dieresis","","ring","cedilla","","hungarumlaut","ogonek","caron","emdash","","","","","","","","","","","","","","","","","AE","","ordfeminine","","","","","Lslash","Oslash","OE","ordmasculine","","","","","","ae","","","","dotlessi","","","lslash","oslash","oe","germandbls"],tt=["","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","space","exclamsmall","Hungarumlautsmall","","dollaroldstyle","dollarsuperior","ampersandsmall","Acutesmall","parenleftsuperior","parenrightsuperior","twodotenleader","onedotenleader","comma","hyphen","period","fraction","zerooldstyle","oneoldstyle","twooldstyle","threeoldstyle","fouroldstyle","fiveoldstyle","sixoldstyle","sevenoldstyle","eightoldstyle","nineoldstyle","colon","semicolon","commasuperior","threequartersemdash","periodsuperior","questionsmall","","asuperior","bsuperior","centsuperior","dsuperior","esuperior","","","isuperior","","","lsuperior","msuperior","nsuperior","osuperior","","","rsuperior","ssuperior","tsuperior","","ff","fi","fl","ffi","ffl","parenleftinferior","","parenrightinferior","Circumflexsmall","hyphensuperior","Gravesmall","Asmall","Bsmall","Csmall","Dsmall","Esmall","Fsmall","Gsmall","Hsmall","Ismall","Jsmall","Ksmall","Lsmall","Msmall","Nsmall","Osmall","Psmall","Qsmall","Rsmall","Ssmall","Tsmall","Usmall","Vsmall","Wsmall","Xsmall","Ysmall","Zsmall","colonmonetary","onefitted","rupiah","Tildesmall","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","exclamdownsmall","centoldstyle","Lslashsmall","","","Scaronsmall","Zcaronsmall","Dieresissmall","Brevesmall","Caronsmall","","Dotaccentsmall","","","Macronsmall","","","figuredash","hypheninferior","","","Ogoneksmall","Ringsmall","Cedillasmall","","","","onequarter","onehalf","threequarters","questiondownsmall","oneeighth","threeeighths","fiveeighths","seveneighths","onethird","twothirds","","","zerosuperior","onesuperior","twosuperior","threesuperior","foursuperior","fivesuperior","sixsuperior","sevensuperior","eightsuperior","ninesuperior","zeroinferior","oneinferior","twoinferior","threeinferior","fourinferior","fiveinferior","sixinferior","seveninferior","eightinferior","nineinferior","centinferior","dollarinferior","periodinferior","commainferior","Agravesmall","Aacutesmall","Acircumflexsmall","Atildesmall","Adieresissmall","Aringsmall","AEsmall","Ccedillasmall","Egravesmall","Eacutesmall","Ecircumflexsmall","Edieresissmall","Igravesmall","Iacutesmall","Icircumflexsmall","Idieresissmall","Ethsmall","Ntildesmall","Ogravesmall","Oacutesmall","Ocircumflexsmall","Otildesmall","Odieresissmall","OEsmall","Oslashsmall","Ugravesmall","Uacutesmall","Ucircumflexsmall","Udieresissmall","Yacutesmall","Thornsmall","Ydieresissmall"],nt=[".notdef",".null","nonmarkingreturn","space","exclam","quotedbl","numbersign","dollar","percent","ampersand","quotesingle","parenleft","parenright","asterisk","plus","comma","hyphen","period","slash","zero","one","two","three","four","five","six","seven","eight","nine","colon","semicolon","less","equal","greater","question","at","A","B","C","D","E","F","G","H","I","J","K","L","M","N","O","P","Q","R","S","T","U","V","W","X","Y","Z","bracketleft","backslash","bracketright","asciicircum","underscore","grave","a","b","c","d","e","f","g","h","i","j","k","l","m","n","o","p","q","r","s","t","u","v","w","x","y","z","braceleft","bar","braceright","asciitilde","Adieresis","Aring","Ccedilla","Eacute","Ntilde","Odieresis","Udieresis","aacute","agrave","acircumflex","adieresis","atilde","aring","ccedilla","eacute","egrave","ecircumflex","edieresis","iacute","igrave","icircumflex","idieresis","ntilde","oacute","ograve","ocircumflex","odieresis","otilde","uacute","ugrave","ucircumflex","udieresis","dagger","degree","cent","sterling","section","bullet","paragraph","germandbls","registered","copyright","trademark","acute","dieresis","notequal","AE","Oslash","infinity","plusminus","lessequal","greaterequal","yen","mu","partialdiff","summation","product","pi","integral","ordfeminine","ordmasculine","Omega","ae","oslash","questiondown","exclamdown","logicalnot","radical","florin","approxequal","Delta","guillemotleft","guillemotright","ellipsis","nonbreakingspace","Agrave","Atilde","Otilde","OE","oe","endash","emdash","quotedblleft","quotedblright","quoteleft","quoteright","divide","lozenge","ydieresis","Ydieresis","fraction","currency","guilsinglleft","guilsinglright","fi","fl","daggerdbl","periodcentered","quotesinglbase","quotedblbase","perthousand","Acircumflex","Ecircumflex","Aacute","Edieresis","Egrave","Iacute","Icircumflex","Idieresis","Igrave","Oacute","Ocircumflex","apple","Ograve","Uacute","Ucircumflex","Ugrave","dotlessi","circumflex","tilde","macron","breve","dotaccent","ring","cedilla","hungarumlaut","ogonek","caron","Lslash","lslash","Scaron","scaron","Zcaron","zcaron","brokenbar","Eth","eth","Yacute","yacute","Thorn","thorn","minus","multiply","onesuperior","twosuperior","threesuperior","onehalf","onequarter","threequarters","franc","Gbreve","gbreve","Idotaccent","Scedilla","scedilla","Cacute","cacute","Ccaron","ccaron","dcroat"];function rt(e){this.font=e}function st(e){this.cmap=e}function ot(e,t){this.encoding=e,this.charset=t}function at(e){switch(e.version){case 1:this.names=nt.slice();break;case 2:this.names=new Array(e.numberOfGlyphs);for(let t=0;t<e.numberOfGlyphs;t++)e.glyphNameIndex[t]<nt.length?this.names[t]=nt[e.glyphNameIndex[t]]:this.names[t]=e.names[e.glyphNameIndex[t]-nt.length];break;case 2.5:this.names=new Array(e.numberOfGlyphs);for(let t=0;t<e.numberOfGlyphs;t++)this.names[t]=nt[t+e.glyphNameIndex[t]];break;default:this.names=[]}}function it(e,t){t.lowMemory?function(e){e._IndexToUnicodeMap={};const t=e.tables.cmap.glyphIndexMap,n=Object.keys(t);for(let r=0;r<n.length;r+=1){const s=n[r];let o=t[s];void 0===e._IndexToUnicodeMap[o]?e._IndexToUnicodeMap[o]={unicodes:[parseInt(s)]}:e._IndexToUnicodeMap[o].unicodes.push(parseInt(s))}}(e):function(e){let t;const n=e.tables.cmap.glyphIndexMap,r=Object.keys(n);for(let s=0;s<r.length;s+=1){const o=r[s],a=n[o];t=e.glyphs.get(a),t.addUnicode(parseInt(o))}for(let n=0;n<e.glyphs.length;n+=1)t=e.glyphs.get(n),e.cffEncoding?t.name=e.cffEncoding.charset[n]:e.glyphNames.names&&(t.name=e.glyphNames.glyphIndexToName(n))}(e)}rt.prototype.charToGlyphIndex=function(e){const t=e.codePointAt(0),n=this.font.glyphs;if(n)for(let e=0;e<n.length;e+=1){const r=n.get(e);for(let n=0;n<r.unicodes.length;n+=1)if(r.unicodes[n]===t)return e}return null},st.prototype.charToGlyphIndex=function(e){return this.cmap.glyphIndexMap[e.codePointAt(0)]||0},ot.prototype.charToGlyphIndex=function(e){const t=e.codePointAt(0),n=this.encoding[t];return this.charset.indexOf(n)},at.prototype.nameToGlyphIndex=function(e){return this.names.indexOf(e)},at.prototype.glyphIndexToName=function(e){return this.names[e]};var lt={line:function(e,t,n,r,s){e.beginPath(),e.moveTo(t,n),e.lineTo(r,s),e.stroke()}};function ut(e){var t=(4278190080&e)>>24,n=(16711680&e)>>16,r=(65280&e)>>8,s=255&e;return{b:t=t+256&255,g:n=n+256&255,r:r=r+256&255,a:s=(s+256&255)/255}}function ct(e,t,n=0,r="hexa"){if(65535==t)return"currentColor";const s=e&&e.tables&&e.tables.cpal;if(!s)return"currentColor";if(n>s.colorRecordIndices.length-1)throw new Error(`Palette index out of range (colorRecordIndices.length: ${s.colorRecordIndices.length}, index: ${t})`);if(t>s.numPaletteEntries)throw new Error(`Color index out of range (numPaletteEntries: ${s.numPaletteEntries}, index: ${t})`);const o=s.colorRecordIndices[n]+t;if(o>s.colorRecords)throw new Error(`Color index out of range (colorRecords.length: ${s.colorRecords.length}, lookupIndex: ${o})`);const a=ut(s.colorRecords[o]);return"bgra"===r?a:dt(a,r)}function pt(e){return("0"+parseInt(e).toString(16)).slice(-2)}function ht(e){return parseInt(`0x${pt(e.b)}${pt(e.g)}${pt(e.r)}${pt(255*e.a)}`,16)}function ft(e,t="hexa"){const n="raw"==t||"cpal"==t,r=Number.isInteger(e);let s=!0;if(r&&n||"currentColor"===e)return e;if("object"==typeof e){if("bgra"==t)return e;if(n)return ht(e)}else if(!r&&/^#([a-f0-9]{3}|[a-f0-9]{4}|[a-f0-9]{6}|[a-f0-9]{8})$/i.test(e.trim())){switch((e=e.trim().substring(1)).length){case 3:e={r:parseInt(e[0].repeat(2),16),g:parseInt(e[1].repeat(2),16),b:parseInt(e[2].repeat(2),16),a:1};break;case 4:e={r:parseInt(e[0].repeat(2),16),g:parseInt(e[1].repeat(2),16),b:parseInt(e[2].repeat(2),16),a:parseInt(e[3].repeat(2),16)/255};break;case 6:e={r:parseInt(e[0]+e[1],16),g:parseInt(e[2]+e[3],16),b:parseInt(e[4]+e[5],16),a:1};break;case 8:e={r:parseInt(e[0]+e[1],16),g:parseInt(e[2]+e[3],16),b:parseInt(e[4]+e[5],16),a:parseInt(e[6]+e[7],16)/255}}if("bgra"==t)return e}else if("undefined"!=typeof document&&/^[a-z]+$/i.test(e)){const t=document.createElement("canvas").getContext("2d");t.fillStyle=e;const n=dt(t.fillStyle,"hexa");"#000000ff"===n&&"black"!==e.toLowerCase()?s=!1:e=n}else{e=e.trim();const t=/rgba?\(\s*(?:(\d*\.\d+)(%?)|(\d+)(%?))\s*(?:,|\s*)\s*(?:(\d*\.\d+)(%?)|(\d+)(%?))\s*(?:,|\s*)\s*(?:(\d*\.\d+)(%?)|(\d+)(%?))\s*(?:(?:,|\s|\/)\s*(?:(0*(?:\.\d+)?()|0*1(?:\.0+)?())|(?:\.\d+)|(\d+)(%)|(\d*\.\d+)(%)))?\s*\)/;if(t.test(e)){const n=e.match(t).filter((e=>void 0!==e));e={r:Math.round(parseFloat(n[1])/(n[2]?100/255:1)),g:Math.round(parseFloat(n[3])/(n[4]?100/255:1)),b:Math.round(parseFloat(n[5])/(n[6]?100/255:1)),a:n[7]?parseFloat(n[7])/(n[8]?100:1):1}}else{const t=/hsla?\(\s*(?:(\d*\.\d+|\d+)(deg|turn|))\s*(?:,|\s*)\s*(?:(\d*\.\d+)%?|(\d+)%?)\s*(?:,|\s*)\s*(?:(\d*\.\d+)%?|(\d+)%?)\s*(?:(?:,|\s|\/)\s*(?:(0*(?:\.\d+)?()|0*1(?:\.0+)?())|(?:\.\d+)|(\d+)(%)|(\d*\.\d+)(%)))?\s*\)/;if(t.test(e)){const n=e.match(t).filter((e=>void 0!==e));e=function(e){let{h:t,s:n,l:r,a:s}=e;t%=360,n/=100,r/=100;const o=(1-Math.abs(2*r-1))*n,a=o*(1-Math.abs(t/60%2-1)),i=r-o/2;let l=0,u=0,c=0;return 0<=t&&t<60?(l=o,u=a,c=0):60<=t&&t<120?(l=a,u=o,c=0):120<=t&&t<180?(l=0,u=o,c=a):180<=t&&t<240?(l=0,u=a,c=o):240<=t&&t<300?(l=a,u=0,c=o):300<=t&&t<=360&&(l=o,u=0,c=a),{r:Math.round(255*(l+i)),g:Math.round(255*(u+i)),b:Math.round(255*(c+i)),a:s}}({h:parseFloat(n[1])*("turn"===n[2]?360:1),s:parseFloat(n[3]),l:parseFloat(n[4]),a:n[5]?parseFloat(n[5])/(n[6]?100:1):1})}else s=!1}}if(!s)throw new Error(`Invalid color format: ${e}`);return dt(e,t)}function dt(e,t="hexa"){if("currentColor"===e)return e;if(Number.isInteger(e)){if("raw"==t||"cpal"==t)return e;e=ut(e)}else"object"!=typeof e&&(e=ft(e,"bgra"));let n=["hsl","hsla"].includes(t)?function(e){const t=e.r/255,n=e.g/255,r=e.b/255,s=Math.max(t,n,r),o=Math.min(t,n,r);let a,i,l=(s+o)/2;if(s===o)a=i=0;else{const e=s-o;switch(i=l>.5?e/(2-s-o):e/(s+o),s){case t:a=(n-r)/e+(n<r?6:0);break;case n:a=(r-t)/e+2;break;case r:a=(t-n)/e+4}a/=6}return{h:360*a,s:100*i,l:100*l}}(e):null;switch(t){case"rgba":return`rgba(${e.r}, ${e.g}, ${e.b}, ${parseFloat(e.a.toFixed(3))})`;case"rgb":return`rgb(${e.r}, ${e.g}, ${e.b})`;case"hex":case"hex6":case"hex-6":return`#${pt(e.r)}${pt(e.g)}${pt(e.b)}`;case"hexa":case"hex8":case"hex-8":return`#${pt(e.r)}${pt(e.g)}${pt(e.b)}${pt(255*e.a)}`;case"hsl":return`hsl(${n.h.toFixed(2)}, ${n.s.toFixed(2)}%, ${n.l.toFixed(2)}%)`;case"hsla":return`hsla(${n.h.toFixed(2)}, ${n.s.toFixed(2)}%, ${n.l.toFixed(2)}%, ${parseFloat(e.a.toFixed(3))})`;case"bgra":return e;case"raw":case"cpal":return ht(e);default:throw new Error("Unknown color format: "+t)}}var gt={parse:function(e,t){const n=new Ce(e,t),r=n.parseShort();0!==r&&console.warn("Only CPALv0 is currently fully supported.");const s=n.parseShort(),o=n.parseShort(),a=n.parseShort(),i=n.parseOffset32(),l=n.parseUShortList(o);n.relativeOffset=i;const u=n.parseULongList(a);return n.relativeOffset=i,{version:r,numPaletteEntries:s,colorRecords:u,colorRecordIndices:l}},make:function({version:e=0,numPaletteEntries:t=0,colorRecords:n=[],colorRecordIndices:r=[0]}){return G.argument(0===e,"Only CPALv0 are supported."),G.argument(n.length,"No colorRecords given."),G.argument(r.length,"No colorRecordIndices given."),r.length>1&&G.argument(t,"Can't infer numPaletteEntries on multiple colorRecordIndices"),new ie.Table("CPAL",[{name:"version",type:"USHORT",value:e},{name:"numPaletteEntries",type:"USHORT",value:t||n.length},{name:"numPalettes",type:"USHORT",value:r.length},{name:"numColorRecords",type:"USHORT",value:n.length},{name:"colorRecordsArrayOffset",type:"ULONG",value:12+2*r.length},...r.map(((e,t)=>({name:"colorRecordIndices_"+t,type:"USHORT",value:e}))),...n.map(((e,t)=>({name:"colorRecords_"+t,type:"ULONG",value:e})))])},getPaletteColor:ct,parseColor:ft,formatColor:dt};function mt(e){this.bindConstructorValues(e)}mt.prototype.bindConstructorValues=function(e){if(this.index=e.index||0,".notdef"===e.name?e.unicode=void 0:".null"===e.name&&(e.unicode=0),0===e.unicode&&".null"!==e.name)throw new Error('The unicode value "0" is reserved for the glyph name ".null" and cannot be used by any other glyph.');this.name=e.name||null,this.unicode=e.unicode,this.unicodes=e.unicodes||(void 0!==e.unicode?[e.unicode]:[]),"xMin"in e&&(this.xMin=e.xMin),"yMin"in e&&(this.yMin=e.yMin),"xMax"in e&&(this.xMax=e.xMax),"yMax"in e&&(this.yMax=e.yMax),"advanceWidth"in e&&(this.advanceWidth=e.advanceWidth),"leftSideBearing"in e&&(this.leftSideBearing=e.leftSideBearing),"points"in e&&(this.points=e.points),Object.defineProperty(this,"path",function(e,t){let n=t||new F;return{configurable:!0,get:function(){return"function"==typeof n&&(n=n()),n},set:function(e){n=e}}}(0,e.path))},mt.prototype.addUnicode=function(e){0===this.unicodes.length&&(this.unicode=e),this.unicodes.push(e)},mt.prototype.getBoundingBox=function(){return this.path.getBoundingBox()},mt.prototype.getPath=function(e,t,n,r,s){let o,a;e=void 0!==e?e:0,t=void 0!==t?t:0,n=void 0!==n?n:72;let i=(r=Object.assign({},s&&s.defaultRenderOptions,r)).xScale,l=r.yScale;const u=1/(this.path.unitsPerEm||1e3)*n;let c=this;s&&s.variation&&(c=s.variation.getTransform(this,r.variation),o=c.path.commands),r.hinting&&s&&s.hinting&&(a=c.path&&s.hinting.exec(c,n,r)),a?(o=s.hinting.getCommands(a),e=Math.round(e),t=Math.round(t),i=l=1):(o=c.path.commands,void 0===i&&(i=u),void 0===l&&(l=u));const p=new F;if(r.drawSVG){const n=this.getSvgImage(s);if(n){const r=new F;return r._image={image:n.image,x:e+n.leftSideBearing*u,y:t-n.baseline*u,width:n.image.width*u,height:n.image.height*u},p._layers=[r],p}}if(r.drawLayers){const o=this.getLayers(s);if(o&&o.length){p._layers=[];for(let a=0;a<o.length;a+=1){const i=o[a];let l=ct(s,i.paletteIndex,r.usePalette);l="currentColor"===l?r.fill||"black":dt(l,r.colorFormat||"rgba"),r=Object.assign({},r,{fill:l}),p._layers.push(this.getPath.call(i.glyph,e,t,n,r,s))}return p}}p.fill=r.fill||this.path.fill,p.stroke=this.path.stroke,p.strokeWidth=this.path.strokeWidth*u;for(let n=0;n<o.length;n+=1){const r=o[n];"M"===r.type?p.moveTo(e+r.x*i,t+-r.y*l):"L"===r.type?p.lineTo(e+r.x*i,t+-r.y*l):"Q"===r.type?p.quadraticCurveTo(e+r.x1*i,t+-r.y1*l,e+r.x*i,t+-r.y*l):"C"===r.type?p.curveTo(e+r.x1*i,t+-r.y1*l,e+r.x2*i,t+-r.y2*l,e+r.x*i,t+-r.y*l):"Z"===r.type&&p.stroke&&p.strokeWidth&&p.closePath()}return p},mt.prototype.getLayers=function(e){if(!e)throw Error("The font object is required to read the colr/cpal tables in order to get the layers.");return e.layers.get(this.index)},mt.prototype.getSvgImage=function(e){if(!e)throw Error("The font object is required to read the svg table in order to get the image.");return e.svgImages.get(this.index)},mt.prototype.getContours=function(e=null){if(void 0===this.points&&!e)return[];const t=[];let n=[],r=e||this.points;for(let e=0;e<r.length;e+=1){const s=r[e];n.push(s),s.lastPointOfContour&&(t.push(n),n=[])}return G.argument(0===n.length,"There are still points left in the current contour."),t},mt.prototype.getMetrics=function(){const e=this.path.commands,t=[],n=[];for(let r=0;r<e.length;r+=1){const s=e[r];"Z"!==s.type&&(t.push(s.x),n.push(s.y)),"Q"!==s.type&&"C"!==s.type||(t.push(s.x1),n.push(s.y1)),"C"===s.type&&(t.push(s.x2),n.push(s.y2))}const r={xMin:Math.min.apply(null,t),yMin:Math.min.apply(null,n),xMax:Math.max.apply(null,t),yMax:Math.max.apply(null,n),leftSideBearing:this.leftSideBearing};return isFinite(r.xMin)||(r.xMin=0),isFinite(r.xMax)||(r.xMax=this.advanceWidth),isFinite(r.yMin)||(r.yMin=0),isFinite(r.yMax)||(r.yMax=0),r.rightSideBearing=this.advanceWidth-r.leftSideBearing-(r.xMax-r.xMin),r},mt.prototype.draw=function(e,t,n,r,s,o){s=Object.assign({},o.defaultRenderOptions,s);this.getPath(t,n,r,s,o).draw(e)},mt.prototype.drawPoints=function(e,t,n,r,s,o){if((s=Object.assign({},o&&o.defaultRenderOptions,s)).drawLayers){const s=this.getLayers(o);if(s&&s.length){for(let o=0;o<s.length;o+=1)s[o].glyph.index!==this.index&&this.drawPoints.call(s[o].glyph,e,t,n,r);return}}function a(t,n,r,s){e.beginPath();for(let o=0;o<t.length;o+=1)e.moveTo(n+t[o].x*s,r+t[o].y*s),e.arc(n+t[o].x*s,r+t[o].y*s,2,0,2*Math.PI,!1);e.fill()}t=void 0!==t?t:0,n=void 0!==n?n:0,r=void 0!==r?r:24;const i=1/this.path.unitsPerEm*r,l=[],u=[];let c=this.path.commands;o&&o.variation&&(c=o.variation.getTransform(this,s.variation).path.commands);for(let e=0;e<c.length;e+=1){const t=c[e];void 0!==t.x&&l.push({x:t.x,y:-t.y}),void 0!==t.x1&&u.push({x:t.x1,y:-t.y1}),void 0!==t.x2&&u.push({x:t.x2,y:-t.y2})}e.fillStyle="blue",a(l,t,n,i),e.fillStyle="red",a(u,t,n,i)},mt.prototype.drawMetrics=function(e,t,n,r){let s;t=void 0!==t?t:0,n=void 0!==n?n:0,r=void 0!==r?r:24,s=1/this.path.unitsPerEm*r,e.lineWidth=1,e.strokeStyle="black",lt.line(e,t,-1e4,t,1e4),lt.line(e,-1e4,n,1e4,n);const o=this.xMin||0;let a=this.yMin||0;const i=this.xMax||0;let l=this.yMax||0;const u=this.advanceWidth||0;e.strokeStyle="blue",lt.line(e,t+o*s,-1e4,t+o*s,1e4),lt.line(e,t+i*s,-1e4,t+i*s,1e4),lt.line(e,-1e4,n+-a*s,1e4,n+-a*s),lt.line(e,-1e4,n+-l*s,1e4,n+-l*s),e.strokeStyle="green",lt.line(e,t+u*s,-1e4,t+u*s,1e4)},mt.prototype.toPathData=function(e,t){e=Object.assign({},{variation:t&&t.defaultRenderOptions.variation},e);let n=this;t&&t.variation&&(n=t.variation.getTransform(this,e.variation));let r=n.points&&e.pointsTransform?e.pointsTransform(n.points):n.path;return e.pathTramsform&&(r=e.pathTramsform(r)),r.toPathData(e)},mt.prototype.fromSVG=function(e,t={}){return this.path.fromSVG(e,t)},mt.prototype.toSVG=function(e,t){const n=this.toPathData.apply(this,[e,t]);return this.path.toSVG(e,n)},mt.prototype.toDOMElement=function(e,t){e=Object.assign({},{variation:t&&t.defaultRenderOptions.variation},e);let n=this.path;return t&&t.variation&&(n=t.variation.getTransform(this,e.variation).path),n.toDOMElement(e)};var yt=mt;function vt(e,t,n){Object.defineProperty(e,t,{get:function(){return void 0===e[n]&&e.path,e[n]},set:function(t){e[n]=t},enumerable:!0,configurable:!0})}function bt(e,t){if(this.font=e,this.glyphs={},Array.isArray(t))for(let n=0;n<t.length;n++){const r=t[n];r.path.unitsPerEm=e.unitsPerEm,this.glyphs[n]=r}this.length=t&&t.length||0}"undefined"!=typeof Symbol&&Symbol.iterator&&(bt.prototype[Symbol.iterator]=function(){let e=-1;return{next:function(){e++;const t=e>=this.length-1;return{value:this.get(e),done:t}}.bind(this)}}),bt.prototype.get=function(e){if(this.font._push&&void 0===this.glyphs[e]){this.font._push(e),"function"==typeof this.glyphs[e]&&(this.glyphs[e]=this.glyphs[e]());let t=this.glyphs[e],n=this.font._IndexToUnicodeMap[e];if(n)for(let e=0;e<n.unicodes.length;e++)t.addUnicode(n.unicodes[e]);this.font.cffEncoding?t.name=this.font.cffEncoding.charset[e]:this.font.glyphNames.names&&(t.name=this.font.glyphNames.glyphIndexToName(e)),this.glyphs[e].advanceWidth=this.font._hmtxTableData[e].advanceWidth,this.glyphs[e].leftSideBearing=this.font._hmtxTableData[e].leftSideBearing}else"function"==typeof this.glyphs[e]&&(this.glyphs[e]=this.glyphs[e]());return this.glyphs[e]},bt.prototype.push=function(e,t){this.glyphs[e]=t,this.length++};var St={GlyphSet:bt,glyphLoader:function(e,t){return new yt({index:t,font:e})},ttfGlyphLoader:function(e,t,n,r,s,o){return function(){const a=new yt({index:t,font:e});return a.path=function(){n(a,r,s);const t=o(e.glyphs,a);return t.unitsPerEm=e.unitsPerEm,t},vt(a,"numberOfContours","_numberOfContours"),vt(a,"xMin","_xMin"),vt(a,"xMax","_xMax"),vt(a,"yMin","_yMin"),vt(a,"yMax","_yMax"),vt(a,"points","_points"),a}},cffGlyphLoader:function(e,t,n,r,s){return function(){const o=new yt({index:t,font:e});return o.path=function(){const t=n(e,o,r,s);return t.unitsPerEm=e.unitsPerEm,t},o}}};function xt(e,t){if(e===t)return!0;if(Array.isArray(e)&&Array.isArray(t)){if(e.length!==t.length)return!1;for(let n=0;n<e.length;n+=1)if(!xt(e[n],t[n]))return!1;return!0}return!1}function Tt(e){let t;return t=e.length<1240?107:e.length<33900?1131:32768,t}function kt(e,t,n,r){const s=[],o=[],a=r>1?Ee.getULong(e,t):Ee.getCard16(e,t),i=r>1?4:2;let l,u;if(0!==a){const n=Ee.getByte(e,t+i);l=t+(a+1)*n+i;let r=t+i+1;for(let t=0;t<a+1;t+=1)s.push(Ee.getOffset(e,r,n)),r+=n;u=l+s[a]}else u=t+i;for(let a=0;a<s.length-1;a+=1){let i=Ee.getBytes(e,l+s[a],l+s[a+1]);n&&(i=n(i,e,t,r)),o.push(i)}return{objects:o,startOffset:t,endOffset:u}}function Ot(e,t){let n,r,s,o;if(28===t)return n=e.parseByte(),r=e.parseByte(),n<<8|r;if(29===t)return n=e.parseByte(),r=e.parseByte(),s=e.parseByte(),o=e.parseByte(),n<<24|r<<16|s<<8|o;if(30===t)return function(e){let t="";const n=["0","1","2","3","4","5","6","7","8","9",".","E","E-",null,"-"];for(;;){const r=e.parseByte(),s=r>>4,o=15&r;if(15===s)break;if(t+=n[s],15===o)break;t+=n[o]}return parseFloat(t)}(e);if(t>=32&&t<=246)return t-139;if(t>=247&&t<=250)return n=e.parseByte(),256*(t-247)+n+108;if(t>=251&&t<=254)return n=e.parseByte(),256*-(t-251)-n-108;throw new Error("Invalid b0 "+t)}function Ut(e,t,n,r){t=void 0!==t?t:0;const s=new Ee.Parser(e,t),o=[];let a=[];n=void 0!==n?n:e.byteLength;let i=r<2?22:28;for(;s.relativeOffset<n;){let e=s.parseByte();if(e<i){if(12===e&&(e=1200+s.parseByte()),r>1&&23===e){Nt(a);continue}o.push([e,a]),a=[]}else a.push(Ot(s,e))}return function(e){const t={};for(let n=0;n<e.length;n+=1){const r=e[n][0],s=e[n][1];let o;if(o=1===s.length?s[0]:s,Object.prototype.hasOwnProperty.call(t,r)&&!isNaN(t[r]))throw new Error("Object "+t+" already has key "+r);t[r]=o}return t}(o)}function Rt(e,t){return t=t<=390?Ze[t]:e?e[t-391]:void 0}function wt(e,t,n){const r={};let s;for(let o=0;o<t.length;o+=1){const a=t[o];if(Array.isArray(a.type)){const t=[];t.length=a.type.length;for(let r=0;r<a.type.length;r++)s=void 0!==e[a.op]?e[a.op][r]:void 0,void 0===s&&(s=void 0!==a.value&&void 0!==a.value[r]?a.value[r]:null),"SID"===a.type[r]&&(s=Rt(n,s)),t[r]=s;r[a.name]=t}else s=e[a.op],void 0===s&&(s=void 0!==a.value?a.value:null),"SID"===a.type&&(s=Rt(n,s)),r[a.name]=s}return r}var It=[{name:"version",op:0,type:"SID"},{name:"notice",op:1,type:"SID"},{name:"copyright",op:1200,type:"SID"},{name:"fullName",op:2,type:"SID"},{name:"familyName",op:3,type:"SID"},{name:"weight",op:4,type:"SID"},{name:"isFixedPitch",op:1201,type:"number",value:0},{name:"italicAngle",op:1202,type:"number",value:0},{name:"underlinePosition",op:1203,type:"number",value:-100},{name:"underlineThickness",op:1204,type:"number",value:50},{name:"paintType",op:1205,type:"number",value:0},{name:"charstringType",op:1206,type:"number",value:2},{name:"fontMatrix",op:1207,type:["real","real","real","real","real","real"],value:[.001,0,0,.001,0,0]},{name:"uniqueId",op:13,type:"number"},{name:"fontBBox",op:5,type:["number","number","number","number"],value:[0,0,0,0]},{name:"strokeWidth",op:1208,type:"number",value:0},{name:"xuid",op:14,type:[],value:null},{name:"charset",op:15,type:"offset",value:0},{name:"encoding",op:16,type:"offset",value:0},{name:"charStrings",op:17,type:"offset",value:0},{name:"private",op:18,type:["number","offset"],value:[0,0]},{name:"ros",op:1230,type:["SID","SID","number"]},{name:"cidFontVersion",op:1231,type:"number",value:0},{name:"cidFontRevision",op:1232,type:"number",value:0},{name:"cidFontType",op:1233,type:"number",value:0},{name:"cidCount",op:1234,type:"number",value:8720},{name:"uidBase",op:1235,type:"number"},{name:"fdArray",op:1236,type:"offset"},{name:"fdSelect",op:1237,type:"offset"},{name:"fontName",op:1238,type:"SID"}],Ct=[{name:"fontMatrix",op:1207,type:["real","real","real","real","real","real"],value:[.001,0,0,.001,0,0]},{name:"charStrings",op:17,type:"offset"},{name:"fdArray",op:1236,type:"offset"},{name:"fdSelect",op:1237,type:"offset"},{name:"vstore",op:24,type:"offset"}],Lt=[{name:"subrs",op:19,type:"offset",value:0},{name:"defaultWidthX",op:20,type:"number",value:0},{name:"nominalWidthX",op:21,type:"number",value:0}],Et=[{name:"blueValues",op:6,type:"delta"},{name:"otherBlues",op:7,type:"delta"},{name:"familyBlues",op:7,type:"delta"},{name:"familyBlues",op:8,type:"delta"},{name:"familyOtherBlues",op:9,type:"delta"},{name:"blueScale",op:1209,type:"number",value:.039625},{name:"blueShift",op:1210,type:"number",value:7},{name:"blueFuzz",op:1211,type:"number",value:1},{name:"stdHW",op:10,type:"number"},{name:"stdVW",op:11,type:"number"},{name:"stemSnapH",op:1212,type:"number"},{name:"stemSnapV",op:1213,type:"number"},{name:"languageGroup",op:1217,type:"number",value:0},{name:"expansionFactor",op:1218,type:"number",value:.06},{name:"vsindex",op:22,type:"number",value:0},{name:"subrs",op:19,type:"offset"}],Dt=[{name:"private",op:18,type:["number","offset"],value:[0,0]}];function Mt(e,t,n,r){return wt(Ut(e,t,e.byteLength,r),r>1?Ct:It,n)}function At(e,t,n,r,s){return wt(Ut(e,t,n,s),s>1?Et:Lt,r)}function Ft(e,t,n){return wt(Ut(e,t,void 0,n),Dt)}function Pt(e,t,n,r,s){const o=[];for(let a=0;a<n.length;a+=1){const i=Mt(new DataView(new Uint8Array(n[a]).buffer),0,r,s);i._subrs=[],i._subrsBias=0,i._defaultWidthX=0,i._nominalWidthX=0;const l=s<2?i.private[0]:0,u=s<2?i.private[1]:0;if(0!==l&&0!==u){const n=At(e,u+t,l,r,s);if(i._defaultWidthX=n.defaultWidthX,i._nominalWidthX=n.nominalWidthX,0!==n.subrs){const r=kt(e,u+n.subrs+t,void 0,s);i._subrs=r.objects,i._subrsBias=Tt(i._subrs)}i._privateDict=n}o.push(i)}return o}function Nt(e){let t=e.pop();for(;e.length>t;)e.pop()}function Gt(e,t){const n=e.tables.cff&&e.tables.cff.topDict&&e.tables.cff.topDict.paintType||0;return 2===n&&(t.fill=null,t.stroke="black",t.strokeWidth=e.tables.cff.topDict.strokeWidth||0),n}function Bt(e,t,n,r,s){let o,a,i,l;const u=new F,c=[];let p,h,f,d,g,m=0,y=!1,v=!1,b=0,S=0,x=0,T=[];const k=e.tables.cff2||e.tables.cff;if(f=k.topDict._defaultWidthX,d=k.topDict._nominalWidthX,s=s||e.variation&&e.variation.get(),t.getBlendPath||(t.getBlendPath=function(s){return Bt(e,t,n,r,s)}),e.isCIDFont||r>1){const e=k.topDict._fdSelect?k.topDict._fdSelect[t.index]:0,n=k.topDict._fdArray[e];p=n._subrs,h=n._subrsBias,r>1?(T=k.topDict._vstore.itemVariationStore,x=n._privateDict.vsindex):(f=n._defaultWidthX,d=n._nominalWidthX)}else p=k.topDict._subrs,h=k.topDict._subrsBias;const O=Gt(e,u);let U=f;function R(e,t){v&&2!==O&&u.closePath(),u.moveTo(e,t),v=!0}function w(){let e;e=0!=(1&c.length),e&&!y&&(U=c.shift()+d),m+=c.length>>1,c.length=0,y=!0}return function n(f){let k,I,C,L,E,D,M,A,F,P,N,G,B=0;for(;B<f.length;){let W=f[B];switch(B+=1,W){case 1:case 3:case 18:case 23:w();break;case 4:c.length>1&&!y&&(U=c.shift()+d,y=!0),S+=c.pop(),R(b,S);break;case 5:for(;c.length>0;)b+=c.shift(),S+=c.shift(),u.lineTo(b,S);break;case 6:for(;c.length>0&&(b+=c.shift(),u.lineTo(b,S),0!==c.length);)S+=c.shift(),u.lineTo(b,S);break;case 7:for(;c.length>0&&(S+=c.shift(),u.lineTo(b,S),0!==c.length);)b+=c.shift(),u.lineTo(b,S);break;case 8:for(;c.length>0;)o=b+c.shift(),a=S+c.shift(),i=o+c.shift(),l=a+c.shift(),b=i+c.shift(),S=l+c.shift(),u.curveTo(o,a,i,l,b,S);break;case 10:E=c.pop()+h,D=p[E],D&&n(D);break;case 11:if(r>1){console.error("CFF CharString operator return (11) is not supported in CFF2");break}return;case 12:switch(W=f[B],B+=1,W){case 35:o=b+c.shift(),a=S+c.shift(),i=o+c.shift(),l=a+c.shift(),M=i+c.shift(),A=l+c.shift(),F=M+c.shift(),P=A+c.shift(),N=F+c.shift(),G=P+c.shift(),b=N+c.shift(),S=G+c.shift(),c.shift(),u.curveTo(o,a,i,l,M,A),u.curveTo(F,P,N,G,b,S);break;case 34:o=b+c.shift(),a=S,i=o+c.shift(),l=a+c.shift(),M=i+c.shift(),A=l,F=M+c.shift(),P=l,N=F+c.shift(),G=S,b=N+c.shift(),u.curveTo(o,a,i,l,M,A),u.curveTo(F,P,N,G,b,S);break;case 36:o=b+c.shift(),a=S+c.shift(),i=o+c.shift(),l=a+c.shift(),M=i+c.shift(),A=l,F=M+c.shift(),P=l,N=F+c.shift(),G=P+c.shift(),b=N+c.shift(),u.curveTo(o,a,i,l,M,A),u.curveTo(F,P,N,G,b,S);break;case 37:o=b+c.shift(),a=S+c.shift(),i=o+c.shift(),l=a+c.shift(),M=i+c.shift(),A=l+c.shift(),F=M+c.shift(),P=A+c.shift(),N=F+c.shift(),G=P+c.shift(),Math.abs(N-b)>Math.abs(G-S)?b=N+c.shift():S=G+c.shift(),u.curveTo(o,a,i,l,M,A),u.curveTo(F,P,N,G,b,S);break;default:console.log("Glyph "+t.index+": unknown operator 1200"+W),c.length=0}break;case 14:if(r>1){console.error("CFF CharString operator endchar (14) is not supported in CFF2");break}if(c.length>=4){const n=et[c.pop()],r=et[c.pop()],s=c.pop(),o=c.pop();if(n&&r){t.isComposite=!0,t.components=[];const a=e.cffEncoding.charset.indexOf(n),i=e.cffEncoding.charset.indexOf(r);t.components.push({glyphIndex:i,dx:0,dy:0}),t.components.push({glyphIndex:a,dx:o,dy:s}),u.extend(e.glyphs.get(i).path);const l=e.glyphs.get(a),c=JSON.parse(JSON.stringify(l.path.commands));for(let e=0;e<c.length;e+=1){const t=c[e];"Z"!==t.type&&(t.x+=o,t.y+=s),"Q"!==t.type&&"C"!==t.type||(t.x1+=o,t.y1+=s),"C"===t.type&&(t.x2+=o,t.y2+=s)}u.extend(c)}}else c.length>0&&!y&&(U=c.shift()+d,y=!0);v&&2!==O&&(u.closePath(),v=!1);break;case 15:if(r<2){console.error("CFF2 CharString operator vsindex (15) is not supported in CFF");break}x=c.pop();break;case 16:if(r<2){console.error("CFF2 CharString operator blend (16) is not supported in CFF");break}g||(g=e.variation&&s&&e.variation.process.getBlendVector(T,x,s));var H=c.pop(),z=g?g.length:T.itemVariationSubtables[x].regionIndexes.length,V=H*z,q=c.length-V,_=q-H;if(g)for(let e=0;e<H;e++){var j=c[_+e];for(let e=0;e<z;e++)j+=g[e]*c[q++];c[_+e]=j}for(;V--;)c.pop();break;case 19:case 20:w(),B+=m+7>>3;break;case 21:c.length>2&&!y&&(U=c.shift()+d,y=!0),S+=c.pop(),b+=c.pop(),R(b,S);break;case 22:c.length>1&&!y&&(U=c.shift()+d,y=!0),b+=c.pop(),R(b,S);break;case 24:for(;c.length>2;)o=b+c.shift(),a=S+c.shift(),i=o+c.shift(),l=a+c.shift(),b=i+c.shift(),S=l+c.shift(),u.curveTo(o,a,i,l,b,S);b+=c.shift(),S+=c.shift(),u.lineTo(b,S);break;case 25:for(;c.length>6;)b+=c.shift(),S+=c.shift(),u.lineTo(b,S);o=b+c.shift(),a=S+c.shift(),i=o+c.shift(),l=a+c.shift(),b=i+c.shift(),S=l+c.shift(),u.curveTo(o,a,i,l,b,S);break;case 26:for(1&c.length&&(b+=c.shift());c.length>0;)o=b,a=S+c.shift(),i=o+c.shift(),l=a+c.shift(),b=i,S=l+c.shift(),u.curveTo(o,a,i,l,b,S);break;case 27:for(1&c.length&&(S+=c.shift());c.length>0;)o=b+c.shift(),a=S,i=o+c.shift(),l=a+c.shift(),b=i+c.shift(),S=l,u.curveTo(o,a,i,l,b,S);break;case 28:k=f[B],I=f[B+1],c.push((k<<24|I<<16)>>16),B+=2;break;case 29:E=c.pop()+e.gsubrsBias,D=e.gsubrs[E],D&&n(D);break;case 30:for(;c.length>0&&(o=b,a=S+c.shift(),i=o+c.shift(),l=a+c.shift(),b=i+c.shift(),S=l+(1===c.length?c.shift():0),u.curveTo(o,a,i,l,b,S),0!==c.length);)o=b+c.shift(),a=S,i=o+c.shift(),l=a+c.shift(),S=l+c.shift(),b=i+(1===c.length?c.shift():0),u.curveTo(o,a,i,l,b,S);break;case 31:for(;c.length>0&&(o=b+c.shift(),a=S,i=o+c.shift(),l=a+c.shift(),S=l+c.shift(),b=i+(1===c.length?c.shift():0),u.curveTo(o,a,i,l,b,S),0!==c.length);)o=b,a=S+c.shift(),i=o+c.shift(),l=a+c.shift(),b=i+c.shift(),S=l+(1===c.length?c.shift():0),u.curveTo(o,a,i,l,b,S);break;default:W<32?console.log("Glyph "+t.index+": unknown operator "+W):W<247?c.push(W-139):W<251?(k=f[B],B+=1,c.push(256*(W-247)+k+108)):W<255?(k=f[B],B+=1,c.push(256*-(W-251)-k-108)):(k=f[B],I=f[B+1],C=f[B+2],L=f[B+3],B+=4,c.push((k<<24|I<<16|C<<8|L)/65536))}}}(n),e.variation&&s&&(u.commands=u.commands.map((e=>{const t=Object.keys(e);for(let n=0;n<t.length;n++){const r=t[n];"type"!==r&&(e[r]=Math.round(e[r]))}return e}))),y&&(t.advanceWidth=U),u}function Ht(e,t,n,r,s){const o=[];let a;const i=new Ee.Parser(e,t),l=i.parseCard8();if(0===l)for(let e=0;e<n;e++){if(a=i.parseCard8(),a>=r)throw new Error("CFF table CID Font FDSelect has bad FD index value "+a+" (FD count "+r+")");o.push(a)}else{if(!(3===l||s>1&&4===l))throw new Error("CFF Table CID Font FDSelect table has unsupported format "+l);{const e=4===l?i.parseULong():i.parseCard16();let t,u=4===l?i.parseULong():i.parseCard16();if(0!==u)throw new Error(`CFF Table CID Font FDSelect format ${l} range has bad initial GID ${u}`);for(let c=0;c<e;c++){if(a=4===l?i.parseUShort():i.parseCard8(),t=4===l?i.parseULong():i.parseCard16(),a>=r)throw new Error("CFF table CID Font FDSelect has bad FD index value "+a+" (FD count "+r+")");if(t>n)throw new Error(`CFF Table CID Font FDSelect format ${s} range has bad GID ${t}`);for(;u<t;u++)o.push(a);u=t}if(t!==n)throw new Error("CFF Table CID Font FDSelect format 3 range has bad final (Sentinal) GID "+t)}}return o}function zt(e,t){let n,r=Ze.indexOf(e);return r>=0&&(n=r),r=t.indexOf(e),r>=0?n=r+Ze.length:(n=Ze.length+t.length,t.push(e)),n}function Vt(e,t,n){const r={};for(let s=0;s<e.length;s+=1){const o=e[s];let a=t[o.name];void 0===a||xt(a,o.value)||("SID"===o.type&&(a=zt(a,n)),r[o.op]={name:o.name,type:o.type,value:a})}return r}function qt(e,t,n){const r=new ie.Record("Top DICT",[{name:"dict",type:"DICT",value:{}}]);return r.dict=Vt(n>1?Ct:It,e,t),r}function _t(e){const t=new ie.Record("Top DICT INDEX",[{name:"topDicts",type:"INDEX",value:[]}]);return t.topDicts=[{name:"topDict_0",type:"TABLE",value:e}],t}function jt(e,t){const n=[],r=e.path;t<2&&n.push({name:"width",type:"NUMBER",value:e.advanceWidth});let s=0,o=0;for(let e=0;e<r.commands.length;e+=1){let t,a,i=r.commands[e];if("Q"===i.type){const e=1/3,t=2/3;i={type:"C",x:i.x,y:i.y,x1:Math.round(e*s+t*i.x1),y1:Math.round(e*o+t*i.y1),x2:Math.round(e*i.x+t*i.x1),y2:Math.round(e*i.y+t*i.y1)}}if("M"===i.type)t=Math.round(i.x-s),a=Math.round(i.y-o),n.push({name:"dx",type:"NUMBER",value:t}),n.push({name:"dy",type:"NUMBER",value:a}),n.push({name:"rmoveto",type:"OP",value:21}),s=Math.round(i.x),o=Math.round(i.y);else if("L"===i.type)t=Math.round(i.x-s),a=Math.round(i.y-o),n.push({name:"dx",type:"NUMBER",value:t}),n.push({name:"dy",type:"NUMBER",value:a}),n.push({name:"rlineto",type:"OP",value:5}),s=Math.round(i.x),o=Math.round(i.y);else if("C"===i.type){const e=Math.round(i.x1-s),r=Math.round(i.y1-o),l=Math.round(i.x2-i.x1),u=Math.round(i.y2-i.y1);t=Math.round(i.x-i.x2),a=Math.round(i.y-i.y2),n.push({name:"dx1",type:"NUMBER",value:e}),n.push({name:"dy1",type:"NUMBER",value:r}),n.push({name:"dx2",type:"NUMBER",value:l}),n.push({name:"dy2",type:"NUMBER",value:u}),n.push({name:"dx",type:"NUMBER",value:t}),n.push({name:"dy",type:"NUMBER",value:a}),n.push({name:"rrcurveto",type:"OP",value:8}),s=Math.round(i.x),o=Math.round(i.y)}}return t<2&&n.push({name:"endchar",type:"OP",value:14}),n}var Wt={parse:function(e,t,n,r){let s;const o=function(e,t){const n={};if(n.formatMajor=Ee.getCard8(e,t),n.formatMinor=Ee.getCard8(e,t+1),n.formatMajor>2)throw new Error(`Unsupported CFF table version ${n.formatMajor}.${n.formatMinor}`);return n.size=Ee.getCard8(e,t+2),n.formatMajor<2?(n.offsetSize=Ee.getCard8(e,t+3),n.startOffset=t,n.endOffset=t+4):(n.topDictLength=Ee.getCard16(e,t+3),n.endOffset=t+8),n}(e,t);s=2===o.formatMajor?n.tables.cff2={}:n.tables.cff={};const a=o.formatMajor>1?null:kt(e,o.endOffset,Ee.bytesToString),i=o.formatMajor>1?null:kt(e,a.endOffset),l=o.formatMajor>1?null:kt(e,i.endOffset,Ee.bytesToString),u=kt(e,o.formatMajor>1?t+o.size+o.topDictLength:l.endOffset,void 0,o.formatMajor);let c,p;if(n.gsubrs=u.objects,n.gsubrsBias=Tt(n.gsubrs),o.formatMajor>1){const n=t+o.size,r=Ee.getBytes(e,n,n+o.topDictLength);c=Pt(e,0,[r],void 0,o.formatMajor)[0]}else{const n=Pt(e,t,i.objects,l.objects,o.formatMajor);if(1!==n.length)throw new Error("CFF table has too many fonts in 'FontSet' - count of fonts NameIndex.length = "+n.length);c=n[0]}if(s.topDict=c,c._privateDict&&(n.defaultWidthX=c._privateDict.defaultWidthX,n.nominalWidthX=c._privateDict.nominalWidthX),o.formatMajor<2&&void 0!==c.ros[0]&&void 0!==c.ros[1]&&(n.isCIDFont=!0),o.formatMajor>1){let r=c.fdArray,s=c.fdSelect;if(!r)throw new Error("This is a CFF2 font, but FDArray information is missing");const a=kt(e,t+r,null,o.formatMajor),i=function(e,t,n){const r=[];for(let s=0;s<n.length;s++){const o=Ft(new DataView(new Uint8Array(n[s]).buffer),0,2),a=o.private[0],i=o.private[1];if(0!==a&&0!==i){const n=At(e,i+t,a,[],2);if(n.subrs){const r=kt(e,i+n.subrs+t,void 0,2);o._subrs=r.objects,o._subrsBias=Tt(o._subrs)}o._privateDict=n}r.push(o)}return r}(e,t,a.objects);c._fdArray=i,s&&(c._fdSelect=Ht(e,t+s,n.numGlyphs,i.length,o.formatMajor))}else if(n.isCIDFont){let r=c.fdArray,s=c.fdSelect;if(0===r||0===s)throw new Error("Font is marked as a CID font, but FDArray and/or FDSelect information is missing");r+=t;const a=kt(e,r),i=Pt(e,t,a.objects,l.objects,o.formatMajor);c._fdArray=i,s+=t,c._fdSelect=Ht(e,s,n.numGlyphs,i.length,o.formatMajor)}if(o.formatMajor<2){const r=t+c.private[1],s=At(e,r,c.private[0],l.objects,o.formatMajor);if(n.defaultWidthX=s.defaultWidthX,n.nominalWidthX=s.nominalWidthX,0!==s.subrs){const t=r+s.subrs,o=kt(e,t);n.subrs=o.objects,n.subrsBias=Tt(n.subrs)}else n.subrs=[],n.subrsBias=0}if(r.lowMemory?(p=function(e,t,n){const r=[],s=n>1?Ee.getULong(e,t):Ee.getCard16(e,t),o=n>1?4:2;let a,i;if(0!==s){const n=Ee.getByte(e,t+o);a=t+(s+1)*n+o;let l=t+o+1;for(let t=0;t<s+1;t+=1)r.push(Ee.getOffset(e,l,n)),l+=n;i=a+r[s]}else i=t+o;return{offsets:r,startOffset:t,endOffset:i}}(e,t+c.charStrings,o.formatMajor),n.nGlyphs=p.offsets.length-(o.formatMajor>1?1:0)):(p=kt(e,t+c.charStrings,null,o.formatMajor),n.nGlyphs=p.objects.length),o.formatMajor>1&&n.tables.maxp&&n.nGlyphs!==n.tables.maxp.numGlyphs&&console.error(`Glyph count in the CFF2 table (${n.nGlyphs}) must correspond to the glyph count in the maxp table (${n.tables.maxp.numGlyphs})`),o.formatMajor<2){let r=[],s=[];r=0===c.charset?Qe:1===c.charset?Ke:2===c.charset?Je:function(e,t,n,r,s){let o,a;const i=new Ee.Parser(e,t);n-=1;const l=[".notdef"],u=i.parseCard8();if(0===u)for(let e=0;e<n;e+=1)o=i.parseSID(),s?l.push(o):l.push(Rt(r,o)||o);else if(1===u)for(;l.length<=n;){o=i.parseSID(),a=i.parseCard8();for(let e=0;e<=a;e+=1)s?l.push("cid"+("00000"+o).slice(-5)):l.push(Rt(r,o)||o),o+=1}else{if(2!==u)throw new Error("Unknown charset format "+u);for(;l.length<=n;){o=i.parseSID(),a=i.parseCard16();for(let e=0;e<=a;e+=1)s?l.push("cid"+("00000"+o).slice(-5)):l.push(Rt(r,o)||o),o+=1}}return l}(e,t+c.charset,n.nGlyphs,l.objects,n.isCIDFont),s=0===c.encoding?et:1===c.encoding?tt:function(e,t){let n;const r={},s=new Ee.Parser(e,t),o=s.parseCard8();if(0===o){const e=s.parseCard8();for(let t=0;t<e;t+=1)n=s.parseCard8(),r[n]=t}else{if(1!==o)throw new Error("Unknown encoding format "+o);{const e=s.parseCard8();n=1;for(let t=0;t<e;t+=1){const e=s.parseCard8(),t=s.parseCard8();for(let s=e;s<=e+t;s+=1)r[s]=n,n+=1}}}return r}(e,t+c.encoding),n.cffEncoding=new ot(s,r),n.encoding=n.encoding||n.cffEncoding}if(n.glyphs=new St.GlyphSet(n),r.lowMemory)n._push=function(r){const s=function(e,t,n,r,s,o){const a=o>1?Ee.getULong(n,r):Ee.getCard16(n,r),i=o>1?4:2;let l=0;0!==a&&(l=r+(a+1)*Ee.getByte(n,r+i)+i);let u=Ee.getBytes(n,l+t[e],l+t[e+1]);return s&&(u=s(u)),u}(r,p.offsets,e,t+c.charStrings,void 0,o.formatMajor);n.glyphs.push(r,St.cffGlyphLoader(n,r,Bt,s,o.formatMajor))};else for(let e=0;e<n.nGlyphs;e+=1){const t=p.objects[e];n.glyphs.push(e,St.cffGlyphLoader(n,e,Bt,t,o.formatMajor))}if(c.vstore){const n=new Ee.Parser(e,t+c.vstore);c._vstore=n.parseVariationStore()}},make:function(e,t){const n=new ie.Table("CFF ",[{name:"header",type:"RECORD"},{name:"nameIndex",type:"RECORD"},{name:"topDictIndex",type:"RECORD"},{name:"stringIndex",type:"RECORD"},{name:"globalSubrIndex",type:"RECORD"},{name:"charsets",type:"RECORD"},{name:"charStringsIndex",type:"RECORD"},{name:"privateDict",type:"RECORD"}]),r=1/t.unitsPerEm,s={version:t.version,fullName:t.fullName,familyName:t.familyName,weight:t.weightName,fontBBox:t.fontBBox||[0,0,0,0],fontMatrix:[r,0,0,r,0,0],charset:999,encoding:0,charStrings:999,private:[0,999]},o=t&&t.topDict||{};o.paintType&&(s.paintType=o.paintType,s.strokeWidth=o.strokeWidth||0);const a=[];let i;for(let t=1;t<e.length;t+=1)i=e.get(t),a.push(i.name);const l=[];n.header=new ie.Record("Header",[{name:"major",type:"Card8",value:1},{name:"minor",type:"Card8",value:0},{name:"hdrSize",type:"Card8",value:4},{name:"major",type:"Card8",value:1}]),n.nameIndex=function(e){const t=new ie.Record("Name INDEX",[{name:"names",type:"INDEX",value:[]}]);t.names=[];for(let n=0;n<e.length;n+=1)t.names.push({name:"name_"+n,type:"NAME",value:e[n]});return t}([t.postScriptName]);let u=qt(s,l);n.topDictIndex=_t(u),n.globalSubrIndex=new ie.Record("Global Subr INDEX",[{name:"subrs",type:"INDEX",value:[]}]),n.charsets=function(e,t){const n=new ie.Record("Charsets",[{name:"format",type:"Card8",value:0}]);for(let r=0;r<e.length;r+=1){const s=zt(e[r],t);n.fields.push({name:"glyph_"+r,type:"SID",value:s})}return n}(a,l),n.charStringsIndex=function(e,t){const n=new ie.Record("CharStrings INDEX",[{name:"charStrings",type:"INDEX",value:[]}]);for(let r=0;r<e.length;r+=1){const s=e.get(r),o=jt(s,t);n.charStrings.push({name:s.name,type:"CHARSTRING",value:o})}return n}(e,1),n.privateDict=function(e,t,n){const r=new ie.Record("Private DICT",[{name:"dict",type:"DICT",value:{}}]);return r.dict=Vt(n>1?Et:Lt,e,t),r}({},l),n.stringIndex=function(e){const t=new ie.Record("String INDEX",[{name:"strings",type:"INDEX",value:[]}]);t.strings=[];for(let n=0;n<e.length;n+=1)t.strings.push({name:"string_"+n,type:"STRING",value:e[n]});return t}(l);const c=n.header.sizeOf()+n.nameIndex.sizeOf()+n.topDictIndex.sizeOf()+n.stringIndex.sizeOf()+n.globalSubrIndex.sizeOf();return s.charset=c,s.encoding=0,s.charStrings=s.charset+n.charsets.sizeOf(),s.private[1]=s.charStrings+n.charStringsIndex.sizeOf(),u=qt(s,l),n.topDictIndex=_t(u),n}};var $t={parse:function(e,t){const n={},r=new Ee.Parser(e,t);return n.version=r.parseVersion(),n.fontRevision=Math.round(1e3*r.parseFixed())/1e3,n.checkSumAdjustment=r.parseULong(),n.magicNumber=r.parseULong(),G.argument(1594834165===n.magicNumber,"Font header has wrong magic number."),n.flags=r.parseUShort(),n.unitsPerEm=r.parseUShort(),n.created=r.parseLongDateTime(),n.modified=r.parseLongDateTime(),n.xMin=r.parseShort(),n.yMin=r.parseShort(),n.xMax=r.parseShort(),n.yMax=r.parseShort(),n.macStyle=r.parseUShort(),n.lowestRecPPEM=r.parseUShort(),n.fontDirectionHint=r.parseShort(),n.indexToLocFormat=r.parseShort(),n.glyphDataFormat=r.parseShort(),n},make:function(e){const t=Math.round((new Date).getTime()/1e3)+2082844800;let n=t,r=e.macStyle||0;return e.createdTimestamp&&(n=e.createdTimestamp+2082844800),new ie.Table("head",[{name:"version",type:"FIXED",value:65536},{name:"fontRevision",type:"FIXED",value:65536},{name:"checkSumAdjustment",type:"ULONG",value:0},{name:"magicNumber",type:"ULONG",value:1594834165},{name:"flags",type:"USHORT",value:0},{name:"unitsPerEm",type:"USHORT",value:1e3},{name:"created",type:"LONGDATETIME",value:n},{name:"modified",type:"LONGDATETIME",value:t},{name:"xMin",type:"SHORT",value:0},{name:"yMin",type:"SHORT",value:0},{name:"xMax",type:"SHORT",value:0},{name:"yMax",type:"SHORT",value:0},{name:"macStyle",type:"USHORT",value:r},{name:"lowestRecPPEM",type:"USHORT",value:0},{name:"fontDirectionHint",type:"SHORT",value:2},{name:"indexToLocFormat",type:"SHORT",value:0},{name:"glyphDataFormat",type:"SHORT",value:0}],e)}};var Yt={parse:function(e,t){const n={},r=new Ee.Parser(e,t);return n.version=r.parseVersion(),n.ascender=r.parseShort(),n.descender=r.parseShort(),n.lineGap=r.parseShort(),n.advanceWidthMax=r.parseUShort(),n.minLeftSideBearing=r.parseShort(),n.minRightSideBearing=r.parseShort(),n.xMaxExtent=r.parseShort(),n.caretSlopeRise=r.parseShort(),n.caretSlopeRun=r.parseShort(),n.caretOffset=r.parseShort(),r.relativeOffset+=8,n.metricDataFormat=r.parseShort(),n.numberOfHMetrics=r.parseUShort(),n},make:function(e){return new ie.Table("hhea",[{name:"version",type:"FIXED",value:65536},{name:"ascender",type:"FWORD",value:0},{name:"descender",type:"FWORD",value:0},{name:"lineGap",type:"FWORD",value:0},{name:"advanceWidthMax",type:"UFWORD",value:0},{name:"minLeftSideBearing",type:"FWORD",value:0},{name:"minRightSideBearing",type:"FWORD",value:0},{name:"xMaxExtent",type:"FWORD",value:0},{name:"caretSlopeRise",type:"SHORT",value:1},{name:"caretSlopeRun",type:"SHORT",value:0},{name:"caretOffset",type:"SHORT",value:0},{name:"reserved1",type:"SHORT",value:0},{name:"reserved2",type:"SHORT",value:0},{name:"reserved3",type:"SHORT",value:0},{name:"reserved4",type:"SHORT",value:0},{name:"metricDataFormat",type:"SHORT",value:0},{name:"numberOfHMetrics",type:"USHORT",value:0}],e)}};var Xt={parse:function(e,t,n,r,s,o,a){a.lowMemory?function(e,t,n,r,s){let o,a;e._hmtxTableData={};const i=new Ee.Parser(t,n);for(let t=0;t<s;t+=1)t<r&&(o=i.parseUShort(),a=i.parseShort()),e._hmtxTableData[t]={advanceWidth:o,leftSideBearing:a}}(e,t,n,r,s):function(e,t,n,r,s){let o,a;const i=new Ee.Parser(e,t);for(let e=0;e<r;e+=1){e<n&&(o=i.parseUShort(),a=i.parseShort());const t=s.get(e);t.advanceWidth=o,t.leftSideBearing=a}}(t,n,r,s,o)},make:function(e){const t=new ie.Table("hmtx",[]);for(let n=0;n<e.length;n+=1){const r=e.get(n),s=r.advanceWidth||0,o=r.leftSideBearing||0;t.fields.push({name:"advanceWidth_"+n,type:"USHORT",value:s}),t.fields.push({name:"leftSideBearing_"+n,type:"SHORT",value:o})}return t}};var Zt={make:function(e){const t=new ie.Table("ltag",[{name:"version",type:"ULONG",value:1},{name:"flags",type:"ULONG",value:0},{name:"numTags",type:"ULONG",value:e.length}]);let n="";const r=12+4*e.length;for(let s=0;s<e.length;++s){let o=n.indexOf(e[s]);o<0&&(o=n.length,n+=e[s]),t.fields.push({name:"offset "+s,type:"USHORT",value:r+o}),t.fields.push({name:"length "+s,type:"USHORT",value:e[s].length})}return t.fields.push({name:"stringPool",type:"CHARARRAY",value:n}),t},parse:function(e,t){const n=new Ee.Parser(e,t),r=n.parseULong();G.argument(1===r,"Unsupported ltag table version."),n.skip("uLong",1);const s=n.parseULong(),o=[];for(let r=0;r<s;r++){let r="";const s=t+n.parseUShort(),a=n.parseUShort();for(let t=s;t<s+a;++t)r+=String.fromCharCode(e.getInt8(t));o.push(r)}return o}};var Qt={parse:function(e,t){const n={},r=new Ee.Parser(e,t);return n.version=r.parseVersion(),n.numGlyphs=r.parseUShort(),1===n.version&&(n.maxPoints=r.parseUShort(),n.maxContours=r.parseUShort(),n.maxCompositePoints=r.parseUShort(),n.maxCompositeContours=r.parseUShort(),n.maxZones=r.parseUShort(),n.maxTwilightPoints=r.parseUShort(),n.maxStorage=r.parseUShort(),n.maxFunctionDefs=r.parseUShort(),n.maxInstructionDefs=r.parseUShort(),n.maxStackElements=r.parseUShort(),n.maxSizeOfInstructions=r.parseUShort(),n.maxComponentElements=r.parseUShort(),n.maxComponentDepth=r.parseUShort()),n},make:function(e){return new ie.Table("maxp",[{name:"version",type:"FIXED",value:20480},{name:"numGlyphs",type:"USHORT",value:e}])}},Kt=[{begin:0,end:127},{begin:128,end:255},{begin:256,end:383},{begin:384,end:591},{begin:592,end:687},{begin:688,end:767},{begin:768,end:879},{begin:880,end:1023},{begin:11392,end:11519},{begin:1024,end:1279},{begin:1328,end:1423},{begin:1424,end:1535},{begin:42240,end:42559},{begin:1536,end:1791},{begin:1984,end:2047},{begin:2304,end:2431},{begin:2432,end:2559},{begin:2560,end:2687},{begin:2688,end:2815},{begin:2816,end:2943},{begin:2944,end:3071},{begin:3072,end:3199},{begin:3200,end:3327},{begin:3328,end:3455},{begin:3584,end:3711},{begin:3712,end:3839},{begin:4256,end:4351},{begin:6912,end:7039},{begin:4352,end:4607},{begin:7680,end:7935},{begin:7936,end:8191},{begin:8192,end:8303},{begin:8304,end:8351},{begin:8352,end:8399},{begin:8400,end:8447},{begin:8448,end:8527},{begin:8528,end:8591},{begin:8592,end:8703},{begin:8704,end:8959},{begin:8960,end:9215},{begin:9216,end:9279},{begin:9280,end:9311},{begin:9312,end:9471},{begin:9472,end:9599},{begin:9600,end:9631},{begin:9632,end:9727},{begin:9728,end:9983},{begin:9984,end:10175},{begin:12288,end:12351},{begin:12352,end:12447},{begin:12448,end:12543},{begin:12544,end:12591},{begin:12592,end:12687},{begin:43072,end:43135},{begin:12800,end:13055},{begin:13056,end:13311},{begin:44032,end:55215},{begin:55296,end:57343},{begin:67840,end:67871},{begin:19968,end:40959},{begin:57344,end:63743},{begin:12736,end:12783},{begin:64256,end:64335},{begin:64336,end:65023},{begin:65056,end:65071},{begin:65040,end:65055},{begin:65104,end:65135},{begin:65136,end:65279},{begin:65280,end:65519},{begin:65520,end:65535},{begin:3840,end:4095},{begin:1792,end:1871},{begin:1920,end:1983},{begin:3456,end:3583},{begin:4096,end:4255},{begin:4608,end:4991},{begin:5024,end:5119},{begin:5120,end:5759},{begin:5760,end:5791},{begin:5792,end:5887},{begin:6016,end:6143},{begin:6144,end:6319},{begin:10240,end:10495},{begin:40960,end:42127},{begin:5888,end:5919},{begin:66304,end:66351},{begin:66352,end:66383},{begin:66560,end:66639},{begin:118784,end:119039},{begin:119808,end:120831},{begin:1044480,end:1048573},{begin:65024,end:65039},{begin:917504,end:917631},{begin:6400,end:6479},{begin:6480,end:6527},{begin:6528,end:6623},{begin:6656,end:6687},{begin:11264,end:11359},{begin:11568,end:11647},{begin:19904,end:19967},{begin:43008,end:43055},{begin:65536,end:65663},{begin:65856,end:65935},{begin:66432,end:66463},{begin:66464,end:66527},{begin:66640,end:66687},{begin:66688,end:66735},{begin:67584,end:67647},{begin:68096,end:68191},{begin:119552,end:119647},{begin:73728,end:74751},{begin:119648,end:119679},{begin:7040,end:7103},{begin:7168,end:7247},{begin:7248,end:7295},{begin:43136,end:43231},{begin:43264,end:43311},{begin:43312,end:43359},{begin:43520,end:43615},{begin:65936,end:65999},{begin:66e3,end:66047},{begin:66208,end:66271},{begin:127024,end:127135}];var Jt={parse:function(e,t){const n={},r=new Ee.Parser(e,t);n.version=r.parseUShort(),n.xAvgCharWidth=r.parseShort(),n.usWeightClass=r.parseUShort(),n.usWidthClass=r.parseUShort(),n.fsType=r.parseUShort(),n.ySubscriptXSize=r.parseShort(),n.ySubscriptYSize=r.parseShort(),n.ySubscriptXOffset=r.parseShort(),n.ySubscriptYOffset=r.parseShort(),n.ySuperscriptXSize=r.parseShort(),n.ySuperscriptYSize=r.parseShort(),n.ySuperscriptXOffset=r.parseShort(),n.ySuperscriptYOffset=r.parseShort(),n.yStrikeoutSize=r.parseShort(),n.yStrikeoutPosition=r.parseShort(),n.sFamilyClass=r.parseShort(),n.panose=[];for(let e=0;e<10;e++)n.panose[e]=r.parseByte();return n.ulUnicodeRange1=r.parseULong(),n.ulUnicodeRange2=r.parseULong(),n.ulUnicodeRange3=r.parseULong(),n.ulUnicodeRange4=r.parseULong(),n.achVendID=String.fromCharCode(r.parseByte(),r.parseByte(),r.parseByte(),r.parseByte()),n.fsSelection=r.parseUShort(),n.usFirstCharIndex=r.parseUShort(),n.usLastCharIndex=r.parseUShort(),n.sTypoAscender=r.parseShort(),n.sTypoDescender=r.parseShort(),n.sTypoLineGap=r.parseShort(),n.usWinAscent=r.parseUShort(),n.usWinDescent=r.parseUShort(),n.version>=1&&(n.ulCodePageRange1=r.parseULong(),n.ulCodePageRange2=r.parseULong()),n.version>=2&&(n.sxHeight=r.parseShort(),n.sCapHeight=r.parseShort(),n.usDefaultChar=r.parseUShort(),n.usBreakChar=r.parseUShort(),n.usMaxContent=r.parseUShort()),n},make:function(e){return new ie.Table("OS/2",[{name:"version",type:"USHORT",value:3},{name:"xAvgCharWidth",type:"SHORT",value:0},{name:"usWeightClass",type:"USHORT",value:0},{name:"usWidthClass",type:"USHORT",value:0},{name:"fsType",type:"USHORT",value:0},{name:"ySubscriptXSize",type:"SHORT",value:650},{name:"ySubscriptYSize",type:"SHORT",value:699},{name:"ySubscriptXOffset",type:"SHORT",value:0},{name:"ySubscriptYOffset",type:"SHORT",value:140},{name:"ySuperscriptXSize",type:"SHORT",value:650},{name:"ySuperscriptYSize",type:"SHORT",value:699},{name:"ySuperscriptXOffset",type:"SHORT",value:0},{name:"ySuperscriptYOffset",type:"SHORT",value:479},{name:"yStrikeoutSize",type:"SHORT",value:49},{name:"yStrikeoutPosition",type:"SHORT",value:258},{name:"sFamilyClass",type:"SHORT",value:0},{name:"bFamilyType",type:"BYTE",value:0},{name:"bSerifStyle",type:"BYTE",value:0},{name:"bWeight",type:"BYTE",value:0},{name:"bProportion",type:"BYTE",value:0},{name:"bContrast",type:"BYTE",value:0},{name:"bStrokeVariation",type:"BYTE",value:0},{name:"bArmStyle",type:"BYTE",value:0},{name:"bLetterform",type:"BYTE",value:0},{name:"bMidline",type:"BYTE",value:0},{name:"bXHeight",type:"BYTE",value:0},{name:"ulUnicodeRange1",type:"ULONG",value:0},{name:"ulUnicodeRange2",type:"ULONG",value:0},{name:"ulUnicodeRange3",type:"ULONG",value:0},{name:"ulUnicodeRange4",type:"ULONG",value:0},{name:"achVendID",type:"CHARARRAY",value:"XXXX"},{name:"fsSelection",type:"USHORT",value:0},{name:"usFirstCharIndex",type:"USHORT",value:0},{name:"usLastCharIndex",type:"USHORT",value:0},{name:"sTypoAscender",type:"SHORT",value:0},{name:"sTypoDescender",type:"SHORT",value:0},{name:"sTypoLineGap",type:"SHORT",value:0},{name:"usWinAscent",type:"USHORT",value:0},{name:"usWinDescent",type:"USHORT",value:0},{name:"ulCodePageRange1",type:"ULONG",value:0},{name:"ulCodePageRange2",type:"ULONG",value:0},{name:"sxHeight",type:"SHORT",value:0},{name:"sCapHeight",type:"SHORT",value:0},{name:"usDefaultChar",type:"USHORT",value:0},{name:"usBreakChar",type:"USHORT",value:0},{name:"usMaxContext",type:"USHORT",value:0}],e)},unicodeRanges:Kt,getUnicodeRange:function(e){for(let t=0;t<Kt.length;t+=1){const n=Kt[t];if(e>=n.begin&&e<n.end)return t}return-1}};var en={parse:function(e,t){const n={},r=new Ee.Parser(e,t);switch(n.version=r.parseVersion(),n.italicAngle=r.parseFixed(),n.underlinePosition=r.parseShort(),n.underlineThickness=r.parseShort(),n.isFixedPitch=r.parseULong(),n.minMemType42=r.parseULong(),n.maxMemType42=r.parseULong(),n.minMemType1=r.parseULong(),n.maxMemType1=r.parseULong(),n.version){case 1:n.names=nt.slice();break;case 2:n.numberOfGlyphs=r.parseUShort(),n.glyphNameIndex=new Array(n.numberOfGlyphs);for(let e=0;e<n.numberOfGlyphs;e++)n.glyphNameIndex[e]=r.parseUShort();n.names=[];for(let e=0;e<n.numberOfGlyphs;e++)if(n.glyphNameIndex[e]>=nt.length){const e=r.parseChar();n.names.push(r.parseString(e))}break;case 2.5:n.numberOfGlyphs=r.parseUShort(),n.offset=new Array(n.numberOfGlyphs);for(let e=0;e<n.numberOfGlyphs;e++)n.offset[e]=r.parseChar()}return n},make:function(e){const{italicAngle:t=Math.round(65536*(e.italicAngle||0)),underlinePosition:n=0,underlineThickness:r=0,isFixedPitch:s=0,minMemType42:o=0,maxMemType42:a=0,minMemType1:i=0,maxMemType1:l=0}=e.tables.post||{};return new ie.Table("post",[{name:"version",type:"FIXED",value:196608},{name:"italicAngle",type:"FIXED",value:t},{name:"underlinePosition",type:"FWORD",value:n},{name:"underlineThickness",type:"FWORD",value:r},{name:"isFixedPitch",type:"ULONG",value:s},{name:"minMemType42",type:"ULONG",value:o},{name:"maxMemType42",type:"ULONG",value:a},{name:"minMemType1",type:"ULONG",value:i},{name:"maxMemType1",type:"ULONG",value:l}])}},tn=new Array(9);tn[1]=function(){const e=this.offset+this.relativeOffset,t=this.parseUShort();return 1===t?{substFormat:1,coverage:this.parsePointer(Ce.coverage),deltaGlyphId:this.parseShort()}:2===t?{substFormat:2,coverage:this.parsePointer(Ce.coverage),substitute:this.parseOffset16List()}:void G.assert(!1,"0x"+e.toString(16)+": lookup type 1 format must be 1 or 2.")},tn[2]=function(){const e=this.parseUShort();return G.argument(1===e,"GSUB Multiple Substitution Subtable identifier-format must be 1"),{substFormat:e,coverage:this.parsePointer(Ce.coverage),sequences:this.parseListOfLists()}},tn[3]=function(){const e=this.parseUShort();return G.argument(1===e,"GSUB Alternate Substitution Subtable identifier-format must be 1"),{substFormat:e,coverage:this.parsePointer(Ce.coverage),alternateSets:this.parseListOfLists()}},tn[4]=function(){const e=this.parseUShort();return G.argument(1===e,"GSUB ligature table identifier-format must be 1"),{substFormat:e,coverage:this.parsePointer(Ce.coverage),ligatureSets:this.parseListOfLists((function(){return{ligGlyph:this.parseUShort(),components:this.parseUShortList(this.parseUShort()-1)}}))}};var nn={sequenceIndex:Ce.uShort,lookupListIndex:Ce.uShort};tn[5]=function(){const e=this.offset+this.relativeOffset,t=this.parseUShort();if(1===t)return{substFormat:t,coverage:this.parsePointer(Ce.coverage),ruleSets:this.parseListOfLists((function(){const e=this.parseUShort(),t=this.parseUShort();return{input:this.parseUShortList(e-1),lookupRecords:this.parseRecordList(t,nn)}}))};if(2===t)return{substFormat:t,coverage:this.parsePointer(Ce.coverage),classDef:this.parsePointer(Ce.classDef),classSets:this.parseListOfLists((function(){const e=this.parseUShort(),t=this.parseUShort();return{classes:this.parseUShortList(e-1),lookupRecords:this.parseRecordList(t,nn)}}))};if(3===t){const e=this.parseUShort(),n=this.parseUShort();return{substFormat:t,coverages:this.parseList(e,Ce.pointer(Ce.coverage)),lookupRecords:this.parseRecordList(n,nn)}}G.assert(!1,"0x"+e.toString(16)+": lookup type 5 format must be 1, 2 or 3.")},tn[6]=function(){const e=this.offset+this.relativeOffset,t=this.parseUShort();return 1===t?{substFormat:1,coverage:this.parsePointer(Ce.coverage),chainRuleSets:this.parseListOfLists((function(){return{backtrack:this.parseUShortList(),input:this.parseUShortList(this.parseShort()-1),lookahead:this.parseUShortList(),lookupRecords:this.parseRecordList(nn)}}))}:2===t?{substFormat:2,coverage:this.parsePointer(Ce.coverage),backtrackClassDef:this.parsePointer(Ce.classDef),inputClassDef:this.parsePointer(Ce.classDef),lookaheadClassDef:this.parsePointer(Ce.classDef),chainClassSet:this.parseListOfLists((function(){return{backtrack:this.parseUShortList(),input:this.parseUShortList(this.parseShort()-1),lookahead:this.parseUShortList(),lookupRecords:this.parseRecordList(nn)}}))}:3===t?{substFormat:3,backtrackCoverage:this.parseList(Ce.pointer(Ce.coverage)),inputCoverage:this.parseList(Ce.pointer(Ce.coverage)),lookaheadCoverage:this.parseList(Ce.pointer(Ce.coverage)),lookupRecords:this.parseRecordList(nn)}:void G.assert(!1,"0x"+e.toString(16)+": lookup type 6 format must be 1, 2 or 3.")},tn[7]=function(){const e=this.parseUShort();G.argument(1===e,"GSUB Extension Substitution subtable identifier-format must be 1");const t=this.parseUShort(),n=new Ce(this.data,this.offset+this.parseULong());return{substFormat:1,lookupType:t,extension:tn[t].call(n)}},tn[8]=function(){const e=this.parseUShort();return G.argument(1===e,"GSUB Reverse Chaining Contextual Single Substitution Subtable identifier-format must be 1"),{substFormat:e,coverage:this.parsePointer(Ce.coverage),backtrackCoverage:this.parseList(Ce.pointer(Ce.coverage)),lookaheadCoverage:this.parseList(Ce.pointer(Ce.coverage)),substitutes:this.parseUShortList()}};var rn=new Array(9);rn[1]=function(e){return 1===e.substFormat?new ie.Table("substitutionTable",[{name:"substFormat",type:"USHORT",value:1},{name:"coverage",type:"TABLE",value:new ie.Coverage(e.coverage)},{name:"deltaGlyphID",type:"SHORT",value:e.deltaGlyphId}]):2===e.substFormat?new ie.Table("substitutionTable",[{name:"substFormat",type:"USHORT",value:2},{name:"coverage",type:"TABLE",value:new ie.Coverage(e.coverage)}].concat(ie.ushortList("substitute",e.substitute))):void G.fail("Lookup type 1 substFormat must be 1 or 2.")},rn[2]=function(e){return G.assert(1===e.substFormat,"Lookup type 2 substFormat must be 1."),new ie.Table("substitutionTable",[{name:"substFormat",type:"USHORT",value:1},{name:"coverage",type:"TABLE",value:new ie.Coverage(e.coverage)}].concat(ie.tableList("seqSet",e.sequences,(function(e){return new ie.Table("sequenceSetTable",ie.ushortList("sequence",e))}))))},rn[3]=function(e){return G.assert(1===e.substFormat,"Lookup type 3 substFormat must be 1."),new ie.Table("substitutionTable",[{name:"substFormat",type:"USHORT",value:1},{name:"coverage",type:"TABLE",value:new ie.Coverage(e.coverage)}].concat(ie.tableList("altSet",e.alternateSets,(function(e){return new ie.Table("alternateSetTable",ie.ushortList("alternate",e))}))))},rn[4]=function(e){return G.assert(1===e.substFormat,"Lookup type 4 substFormat must be 1."),new ie.Table("substitutionTable",[{name:"substFormat",type:"USHORT",value:1},{name:"coverage",type:"TABLE",value:new ie.Coverage(e.coverage)}].concat(ie.tableList("ligSet",e.ligatureSets,(function(e){return new ie.Table("ligatureSetTable",ie.tableList("ligature",e,(function(e){return new ie.Table("ligatureTable",[{name:"ligGlyph",type:"USHORT",value:e.ligGlyph}].concat(ie.ushortList("component",e.components,e.components.length+1)))})))}))))},rn[5]=function(e){if(1===e.substFormat)return new ie.Table("contextualSubstitutionTable",[{name:"substFormat",type:"USHORT",value:e.substFormat},{name:"coverage",type:"TABLE",value:new ie.Coverage(e.coverage)}].concat(ie.tableList("sequenceRuleSet",e.ruleSets,(function(e){return e?new ie.Table("sequenceRuleSetTable",ie.tableList("sequenceRule",e,(function(e){let t=ie.ushortList("seqLookup",[],e.lookupRecords.length).concat(ie.ushortList("inputSequence",e.input,e.input.length+1));[t[0],t[1]]=[t[1],t[0]];for(let n=0;n<e.lookupRecords.length;n++){const r=e.lookupRecords[n];t=t.concat({name:"sequenceIndex"+n,type:"USHORT",value:r.sequenceIndex}).concat({name:"lookupListIndex"+n,type:"USHORT",value:r.lookupListIndex})}return new ie.Table("sequenceRuleTable",t)}))):new ie.Table("NULL",null)}))));if(2===e.substFormat)return new ie.Table("contextualSubstitutionTable",[{name:"substFormat",type:"USHORT",value:e.substFormat},{name:"coverage",type:"TABLE",value:new ie.Coverage(e.coverage)},{name:"classDef",type:"TABLE",value:new ie.ClassDef(e.classDef)}].concat(ie.tableList("classSeqRuleSet",e.classSets,(function(e){return e?new ie.Table("classSeqRuleSetTable",ie.tableList("classSeqRule",e,(function(e){let t=ie.ushortList("classes",e.classes,e.classes.length+1).concat(ie.ushortList("seqLookupCount",[],e.lookupRecords.length));for(let n=0;n<e.lookupRecords.length;n++){const r=e.lookupRecords[n];t=t.concat({name:"sequenceIndex"+n,type:"USHORT",value:r.sequenceIndex}).concat({name:"lookupListIndex"+n,type:"USHORT",value:r.lookupListIndex})}return new ie.Table("classSeqRuleTable",t)}))):new ie.Table("NULL",null)}))));if(3===e.substFormat){let t=[{name:"substFormat",type:"USHORT",value:e.substFormat}];t.push({name:"inputGlyphCount",type:"USHORT",value:e.coverages.length}),t.push({name:"substitutionCount",type:"USHORT",value:e.lookupRecords.length});for(let n=0;n<e.coverages.length;n++){const r=e.coverages[n];t.push({name:"inputCoverage"+n,type:"TABLE",value:new ie.Coverage(r)})}for(let n=0;n<e.lookupRecords.length;n++){const r=e.lookupRecords[n];t=t.concat({name:"sequenceIndex"+n,type:"USHORT",value:r.sequenceIndex}).concat({name:"lookupListIndex"+n,type:"USHORT",value:r.lookupListIndex})}return new ie.Table("contextualSubstitutionTable",t)}G.assert(!1,"lookup type 5 format must be 1, 2 or 3.")},rn[6]=function(e){if(1===e.substFormat){return new ie.Table("chainContextTable",[{name:"substFormat",type:"USHORT",value:e.substFormat},{name:"coverage",type:"TABLE",value:new ie.Coverage(e.coverage)}].concat(ie.tableList("chainRuleSet",e.chainRuleSets,(function(e){return new ie.Table("chainRuleSetTable",ie.tableList("chainRule",e,(function(e){let t=ie.ushortList("backtrackGlyph",e.backtrack,e.backtrack.length).concat(ie.ushortList("inputGlyph",e.input,e.input.length+1)).concat(ie.ushortList("lookaheadGlyph",e.lookahead,e.lookahead.length)).concat(ie.ushortList("substitution",[],e.lookupRecords.length));for(let n=0;n<e.lookupRecords.length;n++){const r=e.lookupRecords[n];t=t.concat({name:"sequenceIndex"+n,type:"USHORT",value:r.sequenceIndex}).concat({name:"lookupListIndex"+n,type:"USHORT",value:r.lookupListIndex})}return new ie.Table("chainRuleTable",t)})))}))))}if(2===e.substFormat)G.assert(!1,"lookup type 6 format 2 is not yet supported.");else if(3===e.substFormat){let t=[{name:"substFormat",type:"USHORT",value:e.substFormat}];t.push({name:"backtrackGlyphCount",type:"USHORT",value:e.backtrackCoverage.length});for(let n=0;n<e.backtrackCoverage.length;n++){const r=e.backtrackCoverage[n];t.push({name:"backtrackCoverage"+n,type:"TABLE",value:new ie.Coverage(r)})}t.push({name:"inputGlyphCount",type:"USHORT",value:e.inputCoverage.length});for(let n=0;n<e.inputCoverage.length;n++){const r=e.inputCoverage[n];t.push({name:"inputCoverage"+n,type:"TABLE",value:new ie.Coverage(r)})}t.push({name:"lookaheadGlyphCount",type:"USHORT",value:e.lookaheadCoverage.length});for(let n=0;n<e.lookaheadCoverage.length;n++){const r=e.lookaheadCoverage[n];t.push({name:"lookaheadCoverage"+n,type:"TABLE",value:new ie.Coverage(r)})}t.push({name:"substitutionCount",type:"USHORT",value:e.lookupRecords.length});for(let n=0;n<e.lookupRecords.length;n++){const r=e.lookupRecords[n];t=t.concat({name:"sequenceIndex"+n,type:"USHORT",value:r.sequenceIndex}).concat({name:"lookupListIndex"+n,type:"USHORT",value:r.lookupListIndex})}return new ie.Table("chainContextTable",t)}G.assert(!1,"lookup type 6 format must be 1, 2 or 3.")};var sn={parse:function(e,t){const n=new Ce(e,t=t||0),r=n.parseVersion(1);return G.argument(1===r||1.1===r,"Unsupported GSUB table version."),1===r?{version:r,scripts:n.parseScriptList(),features:n.parseFeatureList(),lookups:n.parseLookupList(tn)}:{version:r,scripts:n.parseScriptList(),features:n.parseFeatureList(),lookups:n.parseLookupList(tn),variations:n.parseFeatureVariationsList()}},make:function(e){return new ie.Table("GSUB",[{name:"version",type:"ULONG",value:65536},{name:"scripts",type:"TABLE",value:new ie.ScriptList(e.scripts)},{name:"features",type:"TABLE",value:new ie.FeatureList(e.features)},{name:"lookups",type:"TABLE",value:new ie.LookupList(e.lookups,rn)}])}};var on={parse:function(e,t){const n=new Ee.Parser(e,t),r=n.parseULong();G.argument(1===r,"Unsupported META table version."),n.parseULong(),n.parseULong();const s=n.parseULong(),o={};for(let r=0;r<s;r++){const r=n.parseTag(),s=n.parseULong(),a=n.parseULong();if("appl"===r||"bild"===r)continue;const i=H.UTF8(e,t+s,a);o[r]=i}return o},make:function(e){const t=Object.keys(e).length;let n="";const r=16+12*t,s=new ie.Table("meta",[{name:"version",type:"ULONG",value:1},{name:"flags",type:"ULONG",value:0},{name:"offset",type:"ULONG",value:r},{name:"numTags",type:"ULONG",value:t}]);for(let t in e){const o=n.length;n+=e[t],s.fields.push({name:"tag "+t,type:"TAG",value:t}),s.fields.push({name:"offset "+t,type:"ULONG",value:r+o}),s.fields.push({name:"length "+t,type:"ULONG",value:e[t].length})}return s.fields.push({name:"stringPool",type:"CHARARRAY",value:n}),s}};var an={parse:function(e,t){const n=new Ce(e,t),r=n.parseUShort();0!==r&&console.warn("Only COLRv0 is currently fully supported. A subset of color glyphs might be available in this font if provided in the v0 format.");const s=n.parseUShort(),o=n.parseOffset32(),a=n.parseOffset32(),i=n.parseUShort();n.relativeOffset=o;const l=n.parseRecordList(s,{glyphID:Ce.uShort,firstLayerIndex:Ce.uShort,numLayers:Ce.uShort});return n.relativeOffset=a,{version:r,baseGlyphRecords:l,layerRecords:n.parseRecordList(i,{glyphID:Ce.uShort,paletteIndex:Ce.uShort})}},make:function({version:e=0,baseGlyphRecords:t=[],layerRecords:n=[]}){G.argument(0===e,"Only COLRv0 supported.");const r=14+6*t.length;return new ie.Table("COLR",[{name:"version",type:"USHORT",value:e},{name:"numBaseGlyphRecords",type:"USHORT",value:t.length},{name:"baseGlyphRecordsOffset",type:"ULONG",value:14},{name:"layerRecordsOffset",type:"ULONG",value:r},{name:"numLayerRecords",type:"USHORT",value:n.length},...t.map(((e,t)=>[{name:"glyphID_"+t,type:"USHORT",value:e.glyphID},{name:"firstLayerIndex_"+t,type:"USHORT",value:e.firstLayerIndex},{name:"numLayers_"+t,type:"USHORT",value:e.numLayers}])).flat(),...n.map(((e,t)=>[{name:"LayerGlyphID_"+t,type:"USHORT",value:e.glyphID},{name:"paletteIndex_"+t,type:"USHORT",value:e.paletteIndex}])).flat()])}};function ln(e,t,n){const r={},s=new Ee.Parser(e,t);r.tag=s.parseTag(),r.minValue=s.parseFixed(),r.defaultValue=s.parseFixed(),r.maxValue=s.parseFixed(),s.skip("uShort",1);const o=s.parseUShort();return r.axisNameID=o,r.name=We(n,o),r}function un(e,t,n,r={}){const s=[{name:"nameID_"+e,type:"USHORT",value:t.subfamilyNameID},{name:"flags_"+e,type:"USHORT",value:0}];for(let r=0;r<n.length;++r){const o=n[r].tag;s.push({name:"axis_"+e+" "+o,type:"FIXED",value:t.coordinates[o]<<16})}return r&&r.postScriptNameID&&s.push({name:"postScriptNameID_",type:"USHORT",value:void 0!==t.postScriptNameID?t.postScriptNameID:65535}),s}function cn(e,t,n,r,s){const o={},a=new Ee.Parser(e,t),i=a.parseUShort();o.subfamilyNameID=i,o.name=We(r,i,[2,17]),a.skip("uShort",1),o.coordinates={};for(let e=0;e<n.length;++e)o.coordinates[n[e].tag]=a.parseFixed();if(a.relativeOffset===s)return o.postScriptNameID=void 0,o.postScriptName=void 0,o;const l=a.parseUShort();return o.postScriptNameID=65535==l?void 0:l,o.postScriptName=void 0!==o.postScriptNameID?We(r,l,[6]):"",o}var pn={make:function(e,t){const n=new ie.Table("fvar",[{name:"version",type:"ULONG",value:65536},{name:"offsetToData",type:"USHORT",value:0},{name:"countSizePairs",type:"USHORT",value:2},{name:"axisCount",type:"USHORT",value:e.axes.length},{name:"axisSize",type:"USHORT",value:20},{name:"instanceCount",type:"USHORT",value:e.instances.length},{name:"instanceSize",type:"USHORT",value:4+4*e.axes.length}]);n.offsetToData=n.sizeOf();for(let t=0;t<e.axes.length;t++)n.fields=n.fields.concat((r=t,s=e.axes[t],[{name:"tag_"+r,type:"TAG",value:s.tag},{name:"minValue_"+r,type:"FIXED",value:s.minValue<<16},{name:"defaultValue_"+r,type:"FIXED",value:s.defaultValue<<16},{name:"maxValue_"+r,type:"FIXED",value:s.maxValue<<16},{name:"flags_"+r,type:"USHORT",value:0},{name:"nameID_"+r,type:"USHORT",value:s.axisNameID}]));var r,s;const o={};for(let t=0;t<e.instances.length;t++)if(void 0!==e.instances[t].postScriptNameID){n.instanceSize+=2,o.postScriptNameID=!0;break}for(let t=0;t<e.instances.length;t++)n.fields=n.fields.concat(un(t,e.instances[t],e.axes,o));return n},parse:function(e,t,n){const r=new Ee.Parser(e,t),s=r.parseULong();G.argument(65536===s,"Unsupported fvar table version.");const o=r.parseOffset16();r.skip("uShort",1);const a=r.parseUShort(),i=r.parseUShort(),l=r.parseUShort(),u=r.parseUShort(),c=[];for(let r=0;r<a;r++)c.push(ln(e,t+o+r*i,n));const p=[],h=t+o+a*i;for(let t=0;t<l;t++)p.push(cn(e,h+t*u,c,n,u));return{axes:c,instances:p}}},hn={tag:Ce.tag,nameID:Ce.uShort,ordering:Ce.uShort},fn=new Array(5);function dn(){const e=this.parseUShort(),t=fn[e],n={format:e};return void 0===t?(console.warn(`Unknown axis value table format ${e}`),n):Object.assign(n,this.parseStruct(t.bind(this)))}fn[1]=function(){return{axisIndex:this.parseUShort(),flags:this.parseUShort(),valueNameID:this.parseUShort(),value:this.parseFixed()}},fn[2]=function(){return{axisIndex:this.parseUShort(),flags:this.parseUShort(),valueNameID:this.parseUShort(),nominalValue:this.parseFixed(),rangeMinValue:this.parseFixed(),rangeMaxValue:this.parseFixed()}},fn[3]=function(){return{axisIndex:this.parseUShort(),flags:this.parseUShort(),valueNameID:this.parseUShort(),value:this.parseFixed(),linkedValue:this.parseFixed()}},fn[4]=function(){const e=this.parseUShort();return{flags:this.parseUShort(),valueNameID:this.parseUShort(),axisValues:this.parseList(e,(function(){return{axisIndex:this.parseUShort(),value:this.parseFixed()}}))}};var gn=new Array(5);function mn(e,t){const n=t.format,r=gn[n];G.argument(void 0!==r,`Unknown axis value table format ${n}`);const s=r(e,t);return new ie.Table("axisValueTable_"+e,s)}gn[1]=function(e,t){return[{name:`format${e}`,type:"USHORT",value:1},{name:`axisIndex${e}`,type:"USHORT",value:t.axisIndex},{name:`flags${e}`,type:"USHORT",value:t.flags},{name:`valueNameID${e}`,type:"USHORT",value:t.valueNameID},{name:`value${e}`,type:"FLOAT",value:t.value}]},gn[2]=function(e,t){return[{name:`format${e}`,type:"USHORT",value:2},{name:`axisIndex${e}`,type:"USHORT",value:t.axisIndex},{name:`flags${e}`,type:"USHORT",value:t.flags},{name:`valueNameID${e}`,type:"USHORT",value:t.valueNameID},{name:`nominalValue${e}`,type:"FLOAT",value:t.nominalValue},{name:`rangeMinValue${e}`,type:"FLOAT",value:t.rangeMinValue},{name:`rangeMaxValue${e}`,type:"FLOAT",value:t.rangeMaxValue}]},gn[3]=function(e,t){return[{name:`format${e}`,type:"USHORT",value:3},{name:`axisIndex${e}`,type:"USHORT",value:t.axisIndex},{name:`flags${e}`,type:"USHORT",value:t.flags},{name:`valueNameID${e}`,type:"USHORT",value:t.valueNameID},{name:`value${e}`,type:"FLOAT",value:t.value},{name:`linkedValue${e}`,type:"FLOAT",value:t.linkedValue}]},gn[4]=function(e,t){let n=[{name:`format${e}`,type:"USHORT",value:4},{name:`axisCount${e}`,type:"USHORT",value:t.axisValues.length},{name:`flags${e}`,type:"USHORT",value:t.flags},{name:`valueNameID${e}`,type:"USHORT",value:t.valueNameID}];for(let r=0;r<t.axisValues.length;r++)n=n.concat([{name:`format${e}axisIndex${r}`,type:"USHORT",value:t.axisValues[r].axisIndex},{name:`format${e}value${r}`,type:"FLOAT",value:t.axisValues[r].value}]);return n};var yn={make:function(e){const t=new ie.Table("STAT",[{name:"majorVersion",type:"USHORT",value:1},{name:"minorVersion",type:"USHORT",value:2},{name:"designAxisSize",type:"USHORT",value:8},{name:"designAxisCount",type:"USHORT",value:e.axes.length},{name:"designAxesOffset",type:"ULONG",value:0},{name:"axisValueCount",type:"USHORT",value:e.values.length},{name:"offsetToAxisValueOffsets",type:"ULONG",value:0},{name:"elidedFallbackNameID",type:"USHORT",value:e.elidedFallbackNameID}]);t.designAxesOffset=t.offsetToAxisValueOffsets=t.sizeOf();for(let s=0;s<e.axes.length;s++){const o=(n=s,r=e.axes[s],new ie.Record("axisRecord_"+n,[{name:"axisTag_"+n,type:"TAG",value:r.tag},{name:"axisNameID_"+n,type:"USHORT",value:r.nameID},{name:"axisOrdering_"+n,type:"USHORT",value:r.ordering}]));t.offsetToAxisValueOffsets+=o.sizeOf(),t.fields=t.fields.concat(o.fields)}var n,r;const s=[];let o=[],a=2*e.values.length;for(let t=0;t<e.values.length;t++){const n=mn(t,e.values[t]);s.push({name:"offset_"+t,type:"USHORT",value:a}),a+=n.sizeOf(),o=o.concat(n.fields)}return t.fields=t.fields.concat(s),t.fields=t.fields.concat(o),t},parse:function(e,t,n){t||(t=0);const r=new Ee.Parser(e,t),s=r.parseUShort(),o=r.parseUShort();1!==s&&console.warn(`Unsupported STAT table version ${s}.${o}`);const a=[s,o],i=r.parseUShort(),l=r.parseUShort(),u=r.parseOffset32(),c=r.parseUShort(),p=r.parseOffset32(),h=s>1||o>0?r.parseUShort():void 0;void 0!==n&&G.argument(l>=n.axes.length,"STAT axis count must be greater than or equal to fvar axis count"),c>0&&G.argument(l>=0,"STAT axis count must be greater than 0 if STAT axis value count is greater than 0");const f=[];for(let e=0;e<l;e++)r.offset=t+u,r.relativeOffset=e*i,f.push(r.parseStruct(hn));r.offset=t,r.relativeOffset=p;const d=r.parseUShortList(c),g=[];for(let e=0;e<c;e++)r.offset=t+p,r.relativeOffset=d[e],g.push(dn.apply(r));return{version:a,axes:f,values:g,elidedFallbackNameID:h}}};function vn(e,t){return new ie.Record("axisValueMap_"+e,[{name:"fromCoordinate_"+e,type:"F2DOT14",value:t.fromCoordinate},{name:"toCoordinate_"+e,type:"F2DOT14",value:t.toCoordinate}])}function bn(e,t){const n=new ie.Record("segmentMap_"+e,[{name:"positionMapCount_"+e,type:"USHORT",value:t.axisValueMaps.length}]);let r=[];for(let n=0;n<t.axisValueMaps.length;n++){const s=vn(`${e}_${n}`,t.axisValueMaps[n]);r=r.concat(s.fields)}return n.fields=n.fields.concat(r),n}var Sn={make:function(e,t){G.argument(e.axisSegmentMaps.length===t.axes.length,"avar axis count must correspond to fvar axis count");const n=new ie.Table("avar",[{name:"majorVersion",type:"USHORT",value:1},{name:"minorVersion",type:"USHORT",value:0},{name:"reserved",type:"USHORT",value:0},{name:"axisCount",type:"USHORT",value:e.axisSegmentMaps.length}]);for(let t=0;t<e.axisSegmentMaps.length;t++){const r=bn(t,e.axisSegmentMaps[t]);n.fields=n.fields.concat(r.fields)}return n},parse:function(e,t,n){t||(t=0);const r=new Ce(e,t),s=r.parseUShort(),o=r.parseUShort();1!==s&&console.warn(`Unsupported avar table version ${s}.${o}`),r.skip("uShort",1);const a=r.parseUShort();G.argument(a===n.axes.length,"avar axis count must correspond to fvar axis count");const i=[];for(let e=0;e<a;e++){const e=[],t=r.parseUShort();for(let n=0;n<t;n++){const t=r.parseF2Dot14(),n=r.parseF2Dot14();e.push({fromCoordinate:t,toCoordinate:n})}i.push({axisValueMaps:e})}return{version:[s,o],axisSegmentMaps:i}}};var xn={make:function(){console.warn("Writing of cvar tables is not yet supported.")},parse:function(e,t,n,r){const s=new Ee.Parser(e,t),o=s.parseTupleVariationStore(s.relativeOffset,n.axes.length,"cvar",r),a=s.parseUShort(),i=s.parseUShort();return 1!==a&&console.warn(`Unsupported cvar table version ${a}.${i}`),{version:[a,i],...o}}};var Tn={make:function(){console.warn("Writing of gvar tables is not yet supported.")},parse:function(e,t,n,r){const s=new Ee.Parser(e,t),o=s.parseUShort(),a=s.parseUShort();1!==o&&console.warn(`Unsupported gvar table version ${o}.${a}`);const i=s.parseUShort();i!==n.axes.length&&console.warn(`axisCount ${i} in gvar table does not match the number of axes ${n.axes.length} in the fvar table!`);const l=s.parseUShort();return{version:[o,a],sharedTuples:s.parsePointer32((function(){return this.parseTupleRecords(l,i)})),glyphVariations:s.parseTupleVariationStoreList(i,"gvar",r)}}};var kn={parse:function(e,t){const n={},r=new Ee.Parser(e,t);n.version=r.parseUShort(),G.argument(n.version<=1,"Unsupported gasp table version."),n.numRanges=r.parseUShort(),n.gaspRanges=[];for(let e=0;e<n.numRanges;e++)n.gaspRanges[e]={rangeMaxPPEM:r.parseUShort(),rangeGaspBehavior:r.parseUShort()};return n},make:function(e){const t=new ie.Table("gasp",[{name:"version",type:"USHORT",value:1},{name:"numRanges",type:"USHORT",value:e.numRanges}]);for(let n in e.gaspRanges)t.fields.push({name:"rangeMaxPPEM",type:"USHORT",value:e.gaspRanges[n].rangeMaxPPEM}),t.fields.push({name:"rangeGaspBehavior",type:"USHORT",value:e.gaspRanges[n].rangeGaspBehavior});return t}};var On={make:function(e){const t=Array.from(e.keys()).sort(),n=[],r=[],s=new Map;let o=0,a={endGlyphID:null};for(let i=0,l=t.length;i<l;i++){const l=t[i],u=e.get(l);let c=s.get(u);void 0===c&&(c=o,r.push(u),s.set(u,c),o+=u.byteLength),l-1===a.endGlyphID&&c===a.svgDocOffset?a.endGlyphID=l:(a={startGlyphID:l,endGlyphID:l,svgDocOffset:c,svgDocLength:u.byteLength},n.push(a))}const i=n.length,l=r.length,u=2+12*i,c=new Array(4+4*i+l);let p=0;c[p++]={name:"version",type:"USHORT",value:0},c[p++]={name:"svgDocumentListOffset",type:"ULONG",value:10},c[p++]={name:"reserved",type:"ULONG",value:0},c[p++]={name:"numEntries",type:"USHORT",value:i};for(let e=0;e<i;e++){const t="documentRecord_"+e,{startGlyphID:r,endGlyphID:s,svgDocOffset:o,svgDocLength:a}=n[e];c[p++]={name:t+"_startGlyphID",type:"USHORT",value:r},c[p++]={name:t+"_endGlyphID",type:"USHORT",value:s},c[p++]={name:t+"_svgDocOffset",type:"ULONG",value:u+o},c[p++]={name:t+"_svgDocLength",type:"ULONG",value:a}}for(let e=0;e<l;e++)c[p++]={name:"svgDoc_"+e,type:"LITERAL",value:r[e]};return new ie.Table("SVG ",c)},parse:function(e,t){const n=new Map,r=e.buffer,s=new Ce(e,t);if(0!==s.parseUShort())return n;s.relativeOffset=s.parseOffset32();const o=e.byteOffset+t+s.relativeOffset,a=s.parseUShort(),i=new Map;for(let e=0;e<a;e++){const e=s.parseUShort(),t=s.parseUShort(),a=o+s.parseOffset32(),l=s.parseULong();let u=i.get(a);void 0===u&&(u=new Uint8Array(r,a,l),i.set(a,u));for(let r=e;r<=t;r++)n.set(r,u)}return n}};function Un(e){return Math.log(e)/Math.log(2)|0}function Rn(e){for(;e.length%4!=0;)e.push(0);let t=0;for(let n=0;n<e.length;n+=4)t+=(e[n]<<24)+(e[n+1]<<16)+(e[n+2]<<8)+e[n+3];return t%=Math.pow(2,32),t}function wn(e,t,n,r){return new ie.Record("Table Record",[{name:"tag",type:"TAG",value:void 0!==e?e:""},{name:"checkSum",type:"ULONG",value:void 0!==t?t:0},{name:"offset",type:"ULONG",value:void 0!==n?n:0},{name:"length",type:"ULONG",value:void 0!==r?r:0}])}function In(e){const t=new ie.Table("sfnt",[{name:"version",type:"TAG",value:"OTTO"},{name:"numTables",type:"USHORT",value:0},{name:"searchRange",type:"USHORT",value:0},{name:"entrySelector",type:"USHORT",value:0},{name:"rangeShift",type:"USHORT",value:0}]);t.tables=e,t.numTables=e.length;const n=Math.pow(2,Un(t.numTables));t.searchRange=16*n,t.entrySelector=Un(n),t.rangeShift=16*t.numTables-t.searchRange;const r=[],s=[];let o=t.sizeOf()+wn().sizeOf()*t.numTables;for(;o%4!=0;)o+=1,s.push({name:"padding",type:"BYTE",value:0});for(let t=0;t<e.length;t+=1){const n=e[t];G.argument(4===n.tableName.length,"Table name"+n.tableName+" is invalid.");const a=n.sizeOf(),i=wn(n.tableName,Rn(n.encode()),o,a);for(r.push({name:i.tag+" Table Record",type:"RECORD",value:i}),s.push({name:n.tableName+" table",type:"RECORD",value:n}),o+=a,G.argument(!isNaN(o),"Something went wrong calculating the offset.");o%4!=0;)o+=1,s.push({name:"padding",type:"BYTE",value:0})}return r.sort((function(e,t){return e.value.tag>t.value.tag?1:-1})),t.fields=t.fields.concat(r),t.fields=t.fields.concat(s),t}function Cn(e,t,n){for(let n=0;n<t.length;n+=1){const r=e.charToGlyphIndex(t[n]);if(r>0){return e.glyphs.get(r).getMetrics()}}return n}function Ln(e){let t=0;for(let n=0;n<e.length;n+=1)t+=e[n];return t/e.length}var En={make:In,fontToTable:function(e){const t=[],n=[],r=[],s=[],o=[],a=[],i=[];let l,u=0,c=0,p=0,h=0,f=0;for(let d=0;d<e.glyphs.length;d+=1){const g=e.glyphs.get(d),m=0|g.unicode;if(isNaN(g.advanceWidth))throw new Error("Glyph "+g.name+" ("+d+"): advanceWidth is not a number.");(l>m||void 0===l)&&m>0&&(l=m),u<m&&(u=m);const y=Jt.getUnicodeRange(m);if(y<32)c|=1<<y;else if(y<64)p|=1<<y-32;else if(y<96)h|=1<<y-64;else{if(!(y<123))throw new Error("Unicode ranges bits > 123 are reserved for internal usage");f|=1<<y-96}if(".notdef"===g.name)continue;const v=g.getMetrics();t.push(v.xMin),n.push(v.yMin),r.push(v.xMax),s.push(v.yMax),a.push(v.leftSideBearing),i.push(v.rightSideBearing),o.push(g.advanceWidth)}const d={xMin:Math.min.apply(null,t),yMin:Math.min.apply(null,n),xMax:Math.max.apply(null,r),yMax:Math.max.apply(null,s),advanceWidthMax:Math.max.apply(null,o),advanceWidthAvg:Ln(o),minLeftSideBearing:Math.min.apply(null,a),maxLeftSideBearing:Math.max.apply(null,a),minRightSideBearing:Math.min.apply(null,i)};d.ascender=e.ascender,d.descender=e.descender;let g=0;e.weightClass>=600&&(g|=e.macStyleValues.BOLD),e.italicAngle<0&&(g|=e.macStyleValues.ITALIC);const m=$t.make({flags:3,unitsPerEm:e.unitsPerEm,xMin:d.xMin,yMin:d.yMin,xMax:d.xMax,yMax:d.yMax,lowestRecPPEM:3,macStyle:g,createdTimestamp:e.createdTimestamp}),y=Yt.make({ascender:d.ascender,descender:d.descender,advanceWidthMax:d.advanceWidthMax,minLeftSideBearing:d.minLeftSideBearing,minRightSideBearing:d.minRightSideBearing,xMaxExtent:d.maxLeftSideBearing+(d.xMax-d.xMin),numberOfHMetrics:e.glyphs.length}),v=Qt.make(e.glyphs.length),b=Jt.make(Object.assign({xAvgCharWidth:Math.round(d.advanceWidthAvg),usFirstCharIndex:l,usLastCharIndex:u,ulUnicodeRange1:c,ulUnicodeRange2:p,ulUnicodeRange3:h,ulUnicodeRange4:f,sTypoAscender:d.ascender,sTypoDescender:d.descender,sTypoLineGap:0,usWinAscent:d.yMax,usWinDescent:Math.abs(d.yMin),ulCodePageRange1:1,sxHeight:Cn(e,"xyvw",{yMax:Math.round(d.ascender/2)}).yMax,sCapHeight:Cn(e,"HIKLEFJMNTZBDPRAGOQSUVWXY",d).yMax,usDefaultChar:e.hasChar(" ")?32:0,usBreakChar:e.hasChar(" ")?32:0},e.tables.os2)),S=Xt.make(e.glyphs),x=Xe.make(e.glyphs),T=e.getEnglishName("fontFamily"),k=e.getEnglishName("fontSubfamily"),O=T+" "+k;let U=e.getEnglishName("postScriptName");U||(U=T.replace(/\s/g,"")+"-"+k);const R={};for(let t in e.names)R[t]=e.names[t];R.unicode=R.unicode||{},R.macintosh=R.macintosh||{},R.windows=R.windows||{};const w=e.names.unicode||{},I=e.names.macintosh||{},C=e.names.windows||{};for(const t in R){if(R[t]=R[t]||{},!R[t].uniqueID){const n=e.getEnglishName("manufacturer")||"";R[t].uniqueID={en:`${n}: ${O}`}}R[t].postScriptName||(R[t].postScriptName={en:U})}R.unicode.preferredFamily||(R.unicode.preferredFamily=w.fontFamily||I.fontFamily||C.fontFamily),R.macintosh.preferredFamily||(R.macintosh.preferredFamily=I.fontFamily||w.fontFamily||C.fontFamily),R.windows.preferredFamily||(R.windows.preferredFamily=C.fontFamily||w.fontFamily||I.fontFamily),R.unicode.preferredSubfamily||(R.unicode.preferredSubfamily=w.fontSubfamily||I.fontSubfamily||C.fontSubfamily),R.macintosh.preferredSubfamily||(R.macintosh.preferredSubfamily=I.fontSubfamily||w.fontSubfamily||C.fontSubfamily),R.windows.preferredSubfamily||(R.windows.preferredSubfamily=C.fontSubfamily||w.fontSubfamily||I.fontSubfamily);const L=[],E=$e.make(R,L),D=L.length>0?Zt.make(L):void 0,M=en.make(e),A=Wt.make(e.glyphs,{version:e.getEnglishName("version"),fullName:O,familyName:T,weightName:k,postScriptName:U,unitsPerEm:e.unitsPerEm,fontBBox:[0,d.yMin,d.ascender,d.advanceWidthMax],topDict:e.tables.cff&&e.tables.cff.topDict||{}}),F=e.metas&&Object.keys(e.metas).length>0?on.make(e.metas):void 0,P=[m,y,v,b,E,x,M,A,S];D&&P.push(D);const N={gsub:sn,cpal:gt,colr:an,stat:yn,avar:Sn,cvar:xn,fvar:pn,gvar:Tn,gasp:kn,svg:On},G={avar:[e.tables.fvar],fvar:[e.names]};for(let t in N){const n=e.tables[t];if(n){const r=N[t].make.call(e,n,...G[t]||[]);r&&P.push(r)}}F&&P.push(F);const B=In(P),H=Rn(B.encode()),z=B.fields;let V=!1;for(let e=0;e<z.length;e+=1)if("head table"===z[e].name){z[e].value.checkSumAdjustment=2981146554-H,V=!0;break}if(!V)throw new Error("Could not find head table with checkSum to adjust.");return B},computeCheckSum:Rn};function Dn(e,t){let n=0,r=e.length-1;for(;n<=r;){const s=n+r>>>1,o=e[s].tag;if(o===t)return s;o<t?n=s+1:r=s-1}return-n-1}function Mn(e,t){let n=0,r=e.length-1;for(;n<=r;){const s=n+r>>>1,o=e[s];if(o===t)return s;o<t?n=s+1:r=s-1}return-n-1}function An(e,t){let n,r=0,s=e.length-1;for(;r<=s;){const o=r+s>>>1;n=e[o];const a=n.start;if(a===t)return n;a<t?r=o+1:s=o-1}if(r>0)return n=e[r-1],t>n.end?0:n}function Fn(e,t){this.font=e,this.tableName=t}Fn.prototype={searchTag:Dn,binSearch:Mn,getTable:function(e){let t=this.font.tables[this.tableName];return!t&&e&&(t=this.font.tables[this.tableName]=this.createDefaultTable()),t},getScriptNames:function(){let e=this.getTable();return e?e.scripts.map((function(e){return e.tag})):[]},getDefaultScriptName:function(){let e=this.getTable();if(!e)return;let t=!1;for(let n=0;n<e.scripts.length;n++){const r=e.scripts[n].tag;if("DFLT"===r)return r;"latn"===r&&(t=!0)}return t?"latn":void 0},getScriptTable:function(e,t){const n=this.getTable(t);if(n){e=e||"DFLT";const r=n.scripts,s=Dn(n.scripts,e);if(s>=0)return r[s].script;if(t){const t={tag:e,script:{defaultLangSys:{reserved:0,reqFeatureIndex:65535,featureIndexes:[]},langSysRecords:[]}};return r.splice(-1-s,0,t),t.script}}},getLangSysTable:function(e,t,n){const r=this.getScriptTable(e,n);if(r){if(!t||"dflt"===t||"DFLT"===t)return r.defaultLangSys;const e=Dn(r.langSysRecords,t);if(e>=0)return r.langSysRecords[e].langSys;if(n){const n={tag:t,langSys:{reserved:0,reqFeatureIndex:65535,featureIndexes:[]}};return r.langSysRecords.splice(-1-e,0,n),n.langSys}}},getFeatureTable:function(e,t,n,r){const s=this.getLangSysTable(e,t,r);if(s){let e;const t=s.featureIndexes,o=this.font.tables[this.tableName].features;for(let r=0;r<t.length;r++)if(e=o[t[r]],e.tag===n)return e.feature;if(r){const r=o.length;return G.assert(0===r||n>=o[r-1].tag,"Features must be added in alphabetical order."),e={tag:n,feature:{params:0,lookupListIndexes:[]}},o.push(e),t.push(r),e.feature}}},getLookupTables:function(e,t,n,r,s){const o=this.getFeatureTable(e,t,n,s),a=[];if(o){let e;const t=o.lookupListIndexes,n=this.font.tables[this.tableName].lookups;for(let s=0;s<t.length;s++)e=n[t[s]],e.lookupType===r&&a.push(e);if(0===a.length&&s){e={lookupType:r,lookupFlag:0,subtables:[],markFilteringSet:void 0};const s=n.length;return n.push(e),t.push(s),[e]}}return a},getGlyphClass:function(e,t){switch(e.format){case 1:return e.startGlyph<=t&&t<e.startGlyph+e.classes.length?e.classes[t-e.startGlyph]:0;case 2:{const n=An(e.ranges,t);return n?n.classId:0}}},getCoverageIndex:function(e,t){switch(e.format){case 1:{const n=Mn(e.glyphs,t);return n>=0?n:-1}case 2:{const n=An(e.ranges,t);return n?n.index+t-n.start:-1}}},expandCoverage:function(e){if(1===e.format)return e.glyphs;{const t=[],n=e.ranges;for(let e=0;e<n.length;e++){const r=n[e],s=r.start,o=r.end;for(let e=s;e<=o;e++)t.push(e)}return t}}};var Pn=Fn;function Nn(e){Pn.call(this,e,"gpos")}Nn.prototype=Pn.prototype,Nn.prototype.init=function(){const e=this.getDefaultScriptName();this.defaultKerningTables=this.getKerningTables(e)},Nn.prototype.getKerningValue=function(e,t,n){for(let r=0;r<e.length;r++){const s=e[r].subtables;for(let e=0;e<s.length;e++){const r=s[e],o=this.getCoverageIndex(r.coverage,t);if(!(o<0))switch(r.posFormat){case 1:{let e=r.pairSets[o];for(let t=0;t<e.length;t++){let r=e[t];if(r.secondGlyph===n)return r.value1&&r.value1.xAdvance||0}break}case 2:{const e=this.getGlyphClass(r.classDef1,t),s=this.getGlyphClass(r.classDef2,n),o=r.classRecords[e][s];return o.value1&&o.value1.xAdvance||0}}}}return 0},Nn.prototype.getKerningTables=function(e,t){if(this.font.tables.gpos)return this.getLookupTables(e,t,"kern",2)};var Gn=Nn;function Bn(e,t){const n=e.length;if(n!==t.length)return!1;for(let r=0;r<n;r++)if(e[r]!==t[r])return!1;return!0}function Hn(e){return 31===e[0]&&139===e[1]&&8===e[2]}function zn(e){return{x:e.x,y:e.y,onCurve:e.onCurve,lastPointOfContour:e.lastPointOfContour}}function Vn(e){return{glyphIndex:e.glyphIndex,xScale:e.xScale,scale01:e.scale01,scale10:e.scale10,yScale:e.yScale,dx:e.dx,dy:e.dy}}function qn(e){Pn.call(this,e,"gsub")}function _n(e,t,n){const r=e.subtables;for(let e=0;e<r.length;e++){const n=r[e];if(n.substFormat===t)return n}if(n)return r.push(n),n}qn.prototype=Pn.prototype,qn.prototype.createDefaultTable=function(){return{version:1,scripts:[{tag:"DFLT",script:{defaultLangSys:{reserved:0,reqFeatureIndex:65535,featureIndexes:[]},langSysRecords:[]}}],features:[],lookups:[]}},qn.prototype.getSingle=function(e,t,n){const r=[],s=this.getLookupTables(t,n,e,1);for(let e=0;e<s.length;e++){const t=s[e].subtables;for(let e=0;e<t.length;e++){const n=t[e],s=this.expandCoverage(n.coverage);let o;if(1===n.substFormat){const e=n.deltaGlyphId;for(o=0;o<s.length;o++){const t=s[o];r.push({sub:t,by:t+e})}}else{const e=n.substitute;for(o=0;o<s.length;o++)r.push({sub:s[o],by:e[o]})}}}return r},qn.prototype.getMultiple=function(e,t,n){const r=[],s=this.getLookupTables(t,n,e,2);for(let e=0;e<s.length;e++){const t=s[e].subtables;for(let e=0;e<t.length;e++){const n=t[e],s=this.expandCoverage(n.coverage);let o;for(o=0;o<s.length;o++){const e=s[o],t=n.sequences[o];r.push({sub:e,by:t})}}}return r},qn.prototype.getAlternates=function(e,t,n){const r=[],s=this.getLookupTables(t,n,e,3);for(let e=0;e<s.length;e++){const t=s[e].subtables;for(let e=0;e<t.length;e++){const n=t[e],s=this.expandCoverage(n.coverage),o=n.alternateSets;for(let e=0;e<s.length;e++)r.push({sub:s[e],by:o[e]})}}return r},qn.prototype.getLigatures=function(e,t,n){const r=[],s=this.getLookupTables(t,n,e,4);for(let e=0;e<s.length;e++){const t=s[e].subtables;for(let e=0;e<t.length;e++){const n=t[e],s=this.expandCoverage(n.coverage),o=n.ligatureSets;for(let e=0;e<s.length;e++){const t=s[e],n=o[e];for(let e=0;e<n.length;e++){const s=n[e];r.push({sub:[t].concat(s.components),by:s.ligGlyph})}}}}return r},qn.prototype.addSingle=function(e,t,n,r){const s=_n(this.getLookupTables(n,r,e,1,!0)[0],2,{substFormat:2,coverage:{format:1,glyphs:[]},substitute:[]});G.assert(1===s.coverage.format,"Single: unable to modify coverage table format "+s.coverage.format);const o=t.sub;let a=this.binSearch(s.coverage.glyphs,o);a<0&&(a=-1-a,s.coverage.glyphs.splice(a,0,o),s.substitute.splice(a,0,0)),s.substitute[a]=t.by},qn.prototype.addMultiple=function(e,t,n,r){G.assert(t.by instanceof Array&&t.by.length>1,'Multiple: "by" must be an array of two or more ids');const s=_n(this.getLookupTables(n,r,e,2,!0)[0],1,{substFormat:1,coverage:{format:1,glyphs:[]},sequences:[]});G.assert(1===s.coverage.format,"Multiple: unable to modify coverage table format "+s.coverage.format);const o=t.sub;let a=this.binSearch(s.coverage.glyphs,o);a<0&&(a=-1-a,s.coverage.glyphs.splice(a,0,o),s.sequences.splice(a,0,0)),s.sequences[a]=t.by},qn.prototype.addAlternate=function(e,t,n,r){const s=_n(this.getLookupTables(n,r,e,3,!0)[0],1,{substFormat:1,coverage:{format:1,glyphs:[]},alternateSets:[]});G.assert(1===s.coverage.format,"Alternate: unable to modify coverage table format "+s.coverage.format);const o=t.sub;let a=this.binSearch(s.coverage.glyphs,o);a<0&&(a=-1-a,s.coverage.glyphs.splice(a,0,o),s.alternateSets.splice(a,0,0)),s.alternateSets[a]=t.by},qn.prototype.addLigature=function(e,t,n,r){const s=this.getLookupTables(n,r,e,4,!0)[0];let o=s.subtables[0];o||(o={substFormat:1,coverage:{format:1,glyphs:[]},ligatureSets:[]},s.subtables[0]=o),G.assert(1===o.coverage.format,"Ligature: unable to modify coverage table format "+o.coverage.format);const a=t.sub[0],i=t.sub.slice(1),l={ligGlyph:t.by,components:i};let u=this.binSearch(o.coverage.glyphs,a);if(u>=0){const e=o.ligatureSets[u];for(let t=0;t<e.length;t++)if(Bn(e[t].components,i))return;e.push(l)}else u=-1-u,o.coverage.glyphs.splice(u,0,a),o.ligatureSets.splice(u,0,[l])},qn.prototype.getFeature=function(e,t,n){if(/ss\d\d/.test(e))return this.getSingle(e,t,n);switch(e){case"aalt":case"salt":return this.getSingle(e,t,n).concat(this.getAlternates(e,t,n));case"dlig":case"liga":case"rlig":return this.getLigatures(e,t,n);case"ccmp":return this.getMultiple(e,t,n).concat(this.getLigatures(e,t,n));case"stch":return this.getMultiple(e,t,n)}},qn.prototype.add=function(e,t,n,r){if(/ss\d\d/.test(e))return this.addSingle(e,t,n,r);switch(e){case"aalt":case"salt":return"number"==typeof t.by?this.addSingle(e,t,n,r):this.addAlternate(e,t,n,r);case"dlig":case"liga":case"rlig":return this.addLigature(e,t,n,r);case"ccmp":return t.by instanceof Array?this.addMultiple(e,t,n,r):this.addLigature(e,t,n,r)}};var jn=qn,Wn=class{constructor(e){this.defaultValue=255,this.font=e}cpal(){return!(!this.font.tables||!this.font.tables.cpal)&&this.font.tables.cpal}getAll(e){const t=[],n=this.cpal();if(!n)return t;for(let r=0;r<n.colorRecordIndices.length;r++){const s=n.colorRecordIndices[r],o=[];for(let t=s;t<s+n.numPaletteEntries;t++)o.push(dt(n.colorRecords[t],e||"hexa"));t.push(o)}return t}toCPALcolor(e){return Array.isArray(e)?e.map((e=>ft(e,"raw"))):ft(e,"raw")}fillPalette(e,t=[],n=this.cpal().numPaletteEntries){return e=Number.isInteger(e)?this.get(e,"raw"):e,Object.assign(Array(n).fill(this.defaultValue),this.toCPALcolor(e).concat(this.toCPALcolor(t)))}extend(e){if(this.ensureCPAL(Array(e).fill(this.defaultValue)))return;const t=this.cpal(),n=t.numPaletteEntries+e,r=this.getAll().map((e=>this.fillPalette(e,[],n)));t.numPaletteEntries=n,t.colorRecords=this.toCPALcolor(r.flat()),this.updateIndices()}get(e,t="hexa"){return this.getAll(t)[e]||null}getColor(e,t=0,n="hexa"){return ct(this.font,e,t,n)}setColor(e,t,n=0){e=parseInt(e),n=parseInt(n);let r=this.getAll("raw"),s=r[n];if(!s)throw Error(`paletteIndex ${n} out of range`);const o=this.cpal(),a=o.numPaletteEntries;Array.isArray(t)||(t=[t]),t.length+e>a&&(this.extend(t.length+e-a),r=this.getAll("raw"),s=r[n]);for(let n=0;n<t.length;n++)s[n+e]=this.toCPALcolor(t[n]);o.colorRecords=r.flat(),this.updateIndices()}add(e){if(this.ensureCPAL(e))return;const t=this.cpal(),n=t.numPaletteEntries;e&&e.length?((e=this.toCPALcolor(e)).length>n?this.extend(e.length-n):e.length<n&&(e=this.fillPalette(e)),t.colorRecordIndices.push(t.colorRecords.length),t.colorRecords.push(...e)):(t.colorRecordIndices.push(t.colorRecords.length),t.colorRecords.push(...Array(n).fill(this.defaultValue)))}delete(e){const t=this.getAll("raw");delete t[e];const n=this.cpal();n.colorRecordIndices.pop(),n.colorRecords=t.flat()}deleteColor(e,t){if(e===t)throw Error("replacementIndex cannot be the same as colorIndex");const n=this.cpal(),r=this.getAll("raw"),s=[];if(t>n.numPaletteEntries-1)throw Error(`Replacement index out of range: numPaletteEntries after deletion: ${n.numPaletteEntries-1}, replacementIndex: ${t})`);for(let t=0;t<r.length;t++){const n=r[t].filter(((t,n)=>n!==e));s.push(n)}const o=this.font.tables.colr;if(o){const n=o.layerRecords;for(let s=0;s<n.length;s++){const o=n[s].paletteIndex;if(o>e){const e=1;n[s].paletteIndex-=e}else if(o===e){let o=0;for(let n=0;n<r.length;n++)if(t>e&&t<=e+r[n].length){o++;break}n[s].paletteIndex=t-o}}this.font.tables.colr={...o,layerRecords:n}}const a=s.flat();for(let e=0;e<r.length;e++)n.colorRecordIndices[e]-=e;n.numPaletteEntries=Math.max(0,n.numPaletteEntries-1),n.colorRecords=this.toCPALcolor(a)}ensureCPAL(e){return!this.cpal()&&(e=e&&e.length?this.toCPALcolor(e):[this.defaultValue],this.font.tables.cpal={version:0,numPaletteEntries:e.length,colorRecords:e,colorRecordIndices:[0]},!0)}updateIndices(){const e=this.cpal(),t=Math.ceil(e.colorRecords.length/e.numPaletteEntries);e.colorRecordIndices=[];for(let n=0;n<t;n++)e.colorRecordIndices.push(n*e.numPaletteEntries)}},$n=class{constructor(e){this.font=e}ensureCOLR(){return this.font.tables.colr||(this.font.tables.colr={version:0,baseGlyphRecords:[],layerRecords:[]}),this.font}get(e){const t=this.font,n=[],r=t.tables.colr,s=t.tables.cpal;if(!r||!s)return n;const o=function(e,t,n){let r=0,s=e.length-1,o=null;for(;r<=s;){const a=Math.floor((r+s)/2),i=e[a],l=i[t];if(l<n)r=a+1;else{if(!(l>n)){o=i;break}s=a-1}}return o}(r.baseGlyphRecords,"glyphID",e);if(!o)return n;const a=o.firstLayerIndex,i=o.numLayers;for(let e=0;e<i;e++){const s=r.layerRecords[a+e];n.push({glyph:t.glyphs.get(s.glyphID),paletteIndex:s.paletteIndex})}return n}add(e,t,n){const r=this.get(e);t=Array.isArray(t)?t:[t],void 0===n||n===1/0||n>r.length?n=r.length:n<0&&(n=r.length+1+n%(r.length+1))>=r.length+1&&(n-=r.length+1);const s=[];for(let e=0;e<n;e++){const t=Number.isInteger(r[e].glyph)?r[e].glyph:r[e].glyph.index;s.push({glyphID:t,paletteIndex:r[e].paletteIndex})}for(const e of t){const t=Number.isInteger(e.glyph)?e.glyph:e.glyph.index;s.push({glyphID:t,paletteIndex:e.paletteIndex})}for(let e=n;e<r.length;e++){const t=Number.isInteger(r[e].glyph)?r[e].glyph:r[e].glyph.index;s.push({glyphID:t,paletteIndex:r[e].paletteIndex})}this.updateColrTable(e,s)}setPaletteIndex(e,t,n){let r=this.get(e);r[t]?(r=r.map(((e,r)=>({glyphID:e.glyph.index,paletteIndex:r===t?n:e.paletteIndex}))),this.updateColrTable(e,r)):console.error("Invalid layer index")}remove(e,t,n=t){let r=this.get(e);r=r.map((e=>({glyphID:e.glyph.index,paletteIndex:e.paletteIndex}))),r.splice(t,n-t+1),this.updateColrTable(e,r)}updateColrTable(e,t){this.ensureCOLR();const n=this.font.tables.colr;let r=function(e,t,n){let r=0,s=e.length-1;for(;r<=s;){const o=Math.floor((r+s)/2),a=e[o];if(a[t]<n)r=o+1;else{if(!(a[t]>n))return o;s=o-1}}return-1}(n.baseGlyphRecords,"glyphID",e);if(-1===r){const t={glyphID:e,firstLayerIndex:n.layerRecords.length,numLayers:0};r=function(e,t,n){let r=0,s=e.length;for(;r<s;){const a=r+s>>>1;o=n,e[a][t]-o[t]<0?r=a+1:s=a}var o;return e.splice(r,0,n),r}(n.baseGlyphRecords,"glyphID",t)}const s=n.baseGlyphRecords[r],o=s.numLayers,a=t.length,i=a-o;if(i>0){const e=t.slice(o).map((e=>({glyphID:e.glyphID,paletteIndex:e.paletteIndex})));n.layerRecords.splice(s.firstLayerIndex+o,0,...e)}else i<0&&n.layerRecords.splice(s.firstLayerIndex+a,-i);for(let e=0;e<Math.min(o,a);e++)n.layerRecords[s.firstLayerIndex+e]={glyphID:t[e].glyphID,paletteIndex:t[e].paletteIndex};if(s.numLayers=a,0!==i)for(let e=0;e<n.baseGlyphRecords.length;e++){const t=n.baseGlyphRecords[e];e===r||t.firstLayerIndex<s.firstLayerIndex||(n.baseGlyphRecords[e].firstLayerIndex+=i)}}},Yn=class{constructor(e){this.font=e,this.cache=new WeakMap}get(e){const t=this.getOrCreateSvgImageCacheEntry(e);return t&&t.image}getAsync(e){const t=this.getOrCreateSvgImageCacheEntry(e);return t&&t.promise}getOrCreateSvgImageCacheEntry(e){const t=this.font.tables.svg;if(void 0===t)return;const n=t.get(e);if(void 0===n)return;let r=this.cache.get(n);void 0===r&&(r=function(e){return{template:Xn(e).then(Zn),images:new Map}}(n),this.cache.set(n,r));let s=r.images.get(e);return void 0===s&&(s=function(e,t,n){return{promise:t.then((t=>{let r;"string"==typeof t?r=t:(t[4]=n,r=t.join(""));const s=function(e,t){const n=(new DOMParser).parseFromString(e,"image/svg+xml").documentElement,r=n.viewBox.baseVal,s=n.width.baseVal,o=n.height.baseVal;let a=1,i=1;r.width>0&&r.height>0&&(1===s.unitType?(a=s.valueInSpecifiedUnits/r.width,i=1===o.unitType?o.valueInSpecifiedUnits/r.height:a):1===o.unitType?(i=o.valueInSpecifiedUnits/r.height,a=i):t&&(a=t/r.width,i=t/r.height));const l=document.createElement("div");l.style.position="fixed",l.style.visibility="hidden",l.appendChild(n),document.body.appendChild(l);const u=n.getBBox();document.body.removeChild(l);const c=(u.x-r.x)*a,p=(r.y-u.y)*i,h=u.width*a,f=u.height*i;n.setAttribute("viewBox",[u.x,u.y,u.width,u.height].join(" ")),1!==a&&n.setAttribute("width",h);1!==i&&n.setAttribute("height",f);const d=new Image(h,f);return d.src="data:image/svg+xml;charset=utf-8,"+encodeURIComponent(n.outerHTML),{leftSideBearing:c,baseline:p,image:d}}(r,e.unitsPerEm);return s.image.decode().then((()=>s))})),image:void 0}}(this.font,r.template,e),s.promise.then((t=>{if(s.image=t,"function"==typeof this.font.onGlyphUpdated)try{this.font.onGlyphUpdated(e)}catch(t){console.error("font.onGlyphUpdated",e,t)}})),r.images.set(e,s)),s}};var Xn="function"==typeof DecompressionStream?function(e){if(Hn(e))return new Response(new Response(e).body.pipeThrough(new DecompressionStream("gzip"))).text();try{return Promise.resolve((new TextDecoder).decode(e))}catch(e){return Promise.reject(e)}}:function(e){try{return Promise.resolve((new TextDecoder).decode(Hn(e)?function(e){const t=new DataView(e.buffer,e.byteOffset,e.byteLength);let n=10;const r=e.byteLength-8,s=t.getInt8(3);if(4&s&&(n+=2+t.getUint16(n,!0)),8&s)for(;n<r&&0!==e[n++];);if(16&s)for(;n<r&&0!==e[n++];);if(2&s&&(n+=2),n>=r)throw new Error("Can't find compressed blocks");const o=t.getUint32(t.byteLength-4,!0);return w(e.subarray(n,r),new Uint8Array(o))}(e):e))}catch(e){return Promise.reject(e)}};function Zn(e){const t=e.indexOf("<svg"),n=e.indexOf(">",t+4)+1;if(/ id=['"]glyph\d+['"]/.test(e.substring(t,n)))return e;const r=e.lastIndexOf("</svg>");return[e.substring(0,n),"<defs>",e.substring(n,r),'</defs><use href="#glyph',"",'"/>',e.substring(r)]}function Qn(e,t,n,r,s){let o;return(t&r)>0?(o=e.parseByte(),0==(t&s)&&(o=-o),o=n+o):o=(t&s)>0?n:n+e.parseShort(),o}function Kn(e,t,n){const r=new Ee.Parser(t,n);let s,o;if(e._numberOfContours=r.parseShort(),e._xMin=r.parseShort(),e._yMin=r.parseShort(),e._xMax=r.parseShort(),e._yMax=r.parseShort(),e._numberOfContours>0){const t=e.endPointIndices=[];for(let n=0;n<e._numberOfContours;n+=1)t.push(r.parseUShort());e.instructionLength=r.parseUShort(),e.instructions=[];for(let t=0;t<e.instructionLength;t+=1)e.instructions.push(r.parseByte());const n=t[t.length-1]+1;s=[];for(let e=0;e<n;e+=1)if(o=r.parseByte(),s.push(o),(8&o)>0){const t=r.parseByte();for(let n=0;n<t;n+=1)s.push(o),e+=1}if(G.argument(s.length===n,"Bad flags."),t.length>0){const a=[];let i;if(n>0){for(let e=0;e<n;e+=1)o=s[e],i={},i.onCurve=!!(1&o),i.lastPointOfContour=t.indexOf(e)>=0,a.push(i);let e=0;for(let t=0;t<n;t+=1)o=s[t],i=a[t],i.x=Qn(r,o,e,2,16),e=i.x;let l=0;for(let e=0;e<n;e+=1)o=s[e],i=a[e],i.y=Qn(r,o,l,4,32),l=i.y}e.points=a}else e.points=[]}else if(0===e.numberOfContours)e.points=[];else{e.isComposite=!0,e.points=[],e.components=[];let t=!0;for(;t;){s=r.parseUShort();const n={glyphIndex:r.parseUShort(),xScale:1,scale01:0,scale10:0,yScale:1,dx:0,dy:0};(1&s)>0?(2&s)>0?(n.dx=r.parseShort(),n.dy=r.parseShort()):n.matchedPoints=[r.parseUShort(),r.parseUShort()]:(2&s)>0?(n.dx=r.parseChar(),n.dy=r.parseChar()):n.matchedPoints=[r.parseByte(),r.parseByte()],(8&s)>0?n.xScale=n.yScale=r.parseF2Dot14():(64&s)>0?(n.xScale=r.parseF2Dot14(),n.yScale=r.parseF2Dot14()):(128&s)>0&&(n.xScale=r.parseF2Dot14(),n.scale01=r.parseF2Dot14(),n.scale10=r.parseF2Dot14(),n.yScale=r.parseF2Dot14()),e.components.push(n),t=!!(32&s)}if(256&s){e.instructionLength=r.parseUShort(),e.instructions=[];for(let t=0;t<e.instructionLength;t+=1)e.instructions.push(r.parseByte())}}}function Jn(e,t){const n=[];for(let r=0;r<e.length;r+=1){const s=e[r],o={x:t.xScale*s.x+t.scale10*s.y+t.dx,y:t.scale01*s.x+t.yScale*s.y+t.dy,onCurve:s.onCurve,lastPointOfContour:s.lastPointOfContour};n.push(o)}return n}function er(e){const t=new F;if(!e)return t;const n=function(e){const t=[];let n=[];for(let r=0;r<e.length;r+=1){const s=e[r];n.push(s),s.lastPointOfContour&&(t.push(n),n=[])}return G.argument(0===n.length,"There are still points left in the current contour."),t}(e);for(let e=0;e<n.length;++e){const r=n[e];let s=r[r.length-1],o=r[0];if(s.onCurve)t.moveTo(s.x,s.y);else if(o.onCurve)t.moveTo(o.x,o.y);else{const e={x:.5*(s.x+o.x),y:.5*(s.y+o.y)};t.moveTo(e.x,e.y)}for(let e=0;e<r.length;++e)if(s=o,o=r[(e+1)%r.length],s.onCurve)t.lineTo(s.x,s.y);else{let e=o;o.onCurve||(e={x:.5*(s.x+o.x),y:.5*(s.y+o.y)}),t.quadraticCurveTo(s.x,s.y,e.x,e.y)}t.closePath()}return t}function tr(e,t){if(t.isComposite)for(let n=0;n<t.components.length;n+=1){const r=t.components[n],s=e.get(r.glyphIndex);if(s.getPath(),s.points){let e;if(void 0===r.matchedPoints)e=Jn(s.points,r);else{if(r.matchedPoints[0]>t.points.length-1||r.matchedPoints[1]>s.points.length-1)throw Error("Matched points out of range in "+t.name);const n=t.points[r.matchedPoints[0]];let o=s.points[r.matchedPoints[1]];const a={xScale:r.xScale,scale01:r.scale01,scale10:r.scale10,yScale:r.yScale,dx:0,dy:0};o=Jn([o],a)[0],a.dx=n.x-o.x,a.dy=n.y-o.y,e=Jn(s.points,a)}t.points=t.points.concat(e)}}return er(t.points)}var nr,rr,sr,or,ar={getPath:er,parse:function(e,t,n,r,s){return s.lowMemory?function(e,t,n,r){const s=new St.GlyphSet(r);return r._push=function(o){const a=n[o];a!==n[o+1]?s.push(o,St.ttfGlyphLoader(r,o,Kn,e,t+a,tr)):s.push(o,St.glyphLoader(r,o))},s}(e,t,n,r):function(e,t,n,r){const s=new St.GlyphSet(r);for(let o=0;o<n.length-1;o+=1){const a=n[o];a!==n[o+1]?s.push(o,St.ttfGlyphLoader(r,o,Kn,e,t+a,tr)):s.push(o,St.glyphLoader(r,o))}return s}(e,t,n,r)}},ir=class{constructor(e){this.font=e}normalizeCoordTags(e){for(const t in e)if(t.length<4){const n=t.padEnd(4," ");void 0===e[n]&&(e[n]=e[t]),delete e[t]}}getNormalizedCoords(e){e||(e=this.font.variation.get());let t=[];this.normalizeCoordTags(e);for(let n=0;n<this.fvar().axes.length;n++){const r=this.fvar().axes[n];let s=e[r.tag];void 0===s&&(s=r.defaultValue),s<r.defaultValue?t.push((s-r.defaultValue+Number.EPSILON)/(r.defaultValue-r.minValue+Number.EPSILON)):t.push((s-r.defaultValue+Number.EPSILON)/(r.maxValue-r.defaultValue+Number.EPSILON))}if(this.avar())for(let e=0;e<this.avar().axisSegmentMaps.length;e++){let n=this.avar().axisSegmentMaps[e];for(let r=0;r<n.axisValueMaps.length;r++){let s=n.axisValueMaps[r];if(r>=1&&t[e]<s.fromCoordinate){let o=n.axisValueMaps[r-1];t[e]=((t[e]-o.fromCoordinate)*(s.toCoordinate-o.toCoordinate)+Number.EPSILON)/(s.fromCoordinate-o.fromCoordinate+Number.EPSILON)+o.toCoordinate;break}}}return t}interpolatePoints(e,t,n){if(0===e.length)return;let r=0;for(;r<e.length;){let s=r,o=r,a=e[o];for(;!a.lastPointOfContour;)a=e[++o];for(;r<=o&&!n[r];)r++;if(r>o)continue;let i=r,l=r;for(r++;r<=o;)n[r]&&(this.deltaInterpolate(l+1,r-1,l,r,t,e),l=r),r++;l===i?this.deltaShift(s,o,l,t,e):(this.deltaInterpolate(l+1,o,l,i,t,e),i>0&&this.deltaInterpolate(s,i-1,l,i,t,e)),r=o+1}}deltaInterpolate(e,t,n,r,s,o){if(e>t)return;let a=["x","y"];for(let l=0;l<a.length;l++){let u=a[l];if(s[n][u]>s[r][u]){var i=n;n=r,r=i}let c=s[n][u],p=s[r][u],h=o[n][u],f=o[r][u];if(c!==p||h===f){let n=c===p?0:(f-h)/(p-c);for(let r=e;r<=t;r++){let e=s[r][u];e<=c?e+=h-c:e>=p?e+=f-p:e=h+(e-c)*n,o[r][u]=e}}}}deltaShift(e,t,n,r,s){let o=s[n].x-r[n].x,a=s[n].y-r[n].y;if(0!==o||0!==a)for(let r=e;r<=t;r++)r!==n&&(s[r].x+=o,s[r].y+=a)}transformComponents(e,t,n,r,s,o){let a=0;for(let i=0;i<e.components.length;i++){const l=e.components[i],u=this.font.glyphs.get(l.glyphIndex),c=Vn(l),p=r.indexOf(i);p>-1&&(c.dx+=Math.round(s.deltas[p]*o),c.dy+=Math.round(s.deltasY[p]*o));const h=Jn(this.getTransform(u,n).points,c);t.splice(a,h.length,...h),a+=u.points.length}}applyTupleVariationStore(e,t,n,r="gvar",s={}){n||(n=this.font.variation.get());const o=this.getNormalizedCoords(n),{headers:a,sharedPoints:i}=e,l=this.fvar().axes.length;let u;"gvar"===r?u=t.map(zn):"cvar"===r&&(u=[...t]);for(let e=0;e<a.length;e++){const c=a[e];let p=1;for(let e=0;e<l;e++){let t=[0];switch(r){case"gvar":t=c.peakTuple?c.peakTuple:this.gvar().sharedTuples[c.sharedTupleRecordsIndex];break;case"cvar":t=c.peakTuple}if(0!==t[e]){if(0===o[e]){p=0;break}if(c.intermediateStartTuple){if(o[e]<c.intermediateStartTuple[e]||o[e]>c.intermediateEndTuple[e]){p=0;break}p=o[e]<t[e]?p*(o[e]-c.intermediateStartTuple[e]+Number.EPSILON)/(t[e]-c.intermediateStartTuple[e]+Number.EPSILON):p*(c.intermediateEndTuple[e]-o[e]+Number.EPSILON)/(c.intermediateEndTuple[e]-t[e]+Number.EPSILON)}else{if(o[e]<Math.min(0,t[e])||o[e]>Math.max(0,t[e])){p=0;break}p=(p*o[e]+Number.EPSILON)/(t[e]+Number.EPSILON)}}}if(0===p)continue;const h=c.privatePoints.length?c.privatePoints:i;if("gvar"===r&&s.glyph&&s.glyph.isComposite)this.transformComponents(s.glyph,u,n,h,c,p);else if(0===h.length)for(let e=0;e<u.length;e++){const t=u[e];"gvar"===r?u[e]={x:Math.round(t.x+c.deltas[e]*p),y:Math.round(t.y+c.deltasY[e]*p),onCurve:t.onCurve,lastPointOfContour:t.lastPointOfContour}:"cvar"===r&&(u[e]=Math.round(t+c.deltas[e]*p))}else{let e;"gvar"===r?e=u.map(zn):"cvar"===r&&(e=u);const n=Array(t.length).fill(!1);for(let s=0;s<h.length;s++){let o=h[s];if(o<t.length){let t=e[o];"gvar"===r?(n[o]=!0,t.x+=c.deltas[s]*p,t.y+=c.deltasY[s]*p):"cvar"===r&&(u[o]=Math.round(t+c.deltas[s]*p))}}if("gvar"===r){this.interpolatePoints(e,u,n);for(let n=0;n<t.length;n++){let t=e[n].x-u[n].x,r=e[n].y-u[n].y;u[n].x=Math.round(u[n].x+t),u[n].y=Math.round(u[n].y+r)}}}}return u}getTransform(e,t){Number.isInteger(e)&&(e=this.font.glyphs.get(e));const n=e.getBlendPath,r=!(!e.points||!e.points.length);let s=e;if(n||r)if(t||(t=this.font.variation.get()),r){const n=this.gvar()&&this.gvar().glyphVariations[e.index];if(n){const r=e.points;let o=this.applyTupleVariationStore(n,r,t,"gvar",{glyph:e});s=new yt(Object.assign({},e,{points:o,path:er(o)}))}}else if(n){const n=e.getBlendPath(t);s=new yt(Object.assign({},e,{path:n}))}return this.font.tables.hvar&&(e._advanceWidth=void 0!==e._advanceWidth?e._advanceWidth:e.advanceWidth,e.advanceWidth=s.advanceWidth=Math.round(e._advanceWidth+this.getVariableAdjustment(s.index,"hvar","advanceWidth",t)),e._leftSideBearing=void 0!==e._leftSideBearing?e._leftSideBearing:e.leftSideBearing,e.leftSideBearing=s.leftSideBearing=Math.round(e._leftSideBearing+this.getVariableAdjustment(s.index,"hvar","lsb",t))),s}getCvarTransform(e){const t=this.font.tables.cvt,n=this.cvar();return t&&t.length&&n&&n.headers.length?this.applyTupleVariationStore(n,t,e,"cvar"):t}getVariableAdjustment(e,t,n,r){let s,o;r=r||this.font.variation.get();const a=this.font.tables[t];if(!a)throw Error(`trying to get variation adjustment from non-existent table "${a}"`);if(!a.itemVariationStore)throw Error(`trying to get variation adjustment from table "${a}" which does not have an itemVariationStore`);const i=a[n]&&a[n].map.length;if(i){let t=e;t>=i&&(t=i-1),({outerIndex:s,innerIndex:o}=a[n].map[t])}else s=0,o=e;return this.getDelta(a.itemVariationStore,s,o,r)}getDelta(e,t,n,r){if(t>=e.itemVariationSubtables.length)return 0;let s=e.itemVariationSubtables[t];if(n>=s.deltaSets.length)return 0;let o=s.deltaSets[n],a=this.getBlendVector(e,t,r),i=0;for(let e=0;e<s.regionIndexes.length;e++)i+=o[e]*a[e];return i}getBlendVector(e,t,n){n||(n=this.font.variation.get());let r=e.itemVariationSubtables[t];const s=this.getNormalizedCoords(n);let o=[];for(let t=0;t<r.regionIndexes.length;t++){let n=1,a=r.regionIndexes[t],i=e.variationRegions[a].regionAxes;for(let e=0;e<i.length;e++){let t,r=i[e];t=r.startCoord>r.peakCoord||r.peakCoord>r.endCoord||r.startCoord<0&&r.endCoord>0&&0!==r.peakCoord||0===r.peakCoord?1:s[e]<r.startCoord||s[e]>r.endCoord?0:s[e]===r.peakCoord?1:s[e]<r.peakCoord?(s[e]-r.startCoord+Number.EPSILON)/(r.peakCoord-r.startCoord+Number.EPSILON):(r.endCoord-s[e]+Number.EPSILON)/(r.endCoord-r.peakCoord+Number.EPSILON),n*=t}o[t]=n}return o}avar(){return this.font.tables.avar}cvar(){return this.font.tables.cvar}fvar(){return this.font.tables.fvar}gvar(){return this.font.tables.gvar}hvar(){return this.font.tables.hvar}},lr=class{constructor(e){this.font=e,this.process=new ir(this.font),this.activateDefaultVariation(),this.getTransform=this.process.getTransform.bind(this.process)}activateDefaultVariation(){const e=this.getDefaultInstanceIndex();e>-1?this.set(e):this.set(this.getDefaultCoordinates())}getDefaultCoordinates(){return this.fvar().axes.reduce(((e,t)=>(e[t.tag]=t.defaultValue,e)),{})}getDefaultInstanceIndex(){const e=this.getDefaultCoordinates();let t=this.getInstanceIndex(e);return t<0&&(t=this.fvar().instances.findIndex((e=>e.name&&"Regular"===e.name.en))),t}getInstanceIndex(e){return this.fvar().instances.findIndex((t=>Object.keys(e).every((n=>t.coordinates[n]===e[n]))))}getInstance(e){return this.fvar().instances&&this.fvar().instances[e]}set(e){let t;if(Number.isInteger(e)){const n=this.getInstance(e);if(!n)throw Error(`Invalid instance index ${e}`);t={...n.coordinates}}else t=e,this.process.normalizeCoordTags(t);t=Object.assign({},this.font.defaultRenderOptions.variation,t),this.font.defaultRenderOptions=Object.assign({},this.font.defaultRenderOptions,{variation:t})}get(){return Object.assign({},this.font.defaultRenderOptions.variation)}avar(){return this.font.tables.avar}cvar(){return this.font.tables.cvar}fvar(){return this.font.tables.fvar}gvar(){return this.font.tables.gvar}hvar(){return this.font.tables.hvar}};function ur(e){this.font=e,this.getCommands=function(e){return ar.getPath(e).commands},this._fpgmState=this._prepState=void 0,this._errorState=0}function cr(e){return e}function pr(e){return Math.sign(e)*Math.round(Math.abs(e))}function hr(e){return Math.sign(e)*Math.round(Math.abs(2*e))/2}function fr(e){return Math.sign(e)*(Math.round(Math.abs(e)+.5)-.5)}function dr(e){return Math.sign(e)*Math.ceil(Math.abs(e))}function gr(e){return Math.sign(e)*Math.floor(Math.abs(e))}var mr=function(e){const t=this.srPeriod;let n=this.srPhase;let r=1;return e<0&&(e=-e,r=-1),e+=this.srThreshold-n,e=Math.trunc(e/t)*t,(e+=n)<0?n*r:e*r},yr={x:1,y:0,axis:"x",distance:function(e,t,n,r){return(n?e.xo:e.x)-(r?t.xo:t.x)},interpolate:function(e,t,n,r){let s,o,a,i,l,u,c;if(!r||r===this)return s=e.xo-t.xo,o=e.xo-n.xo,l=t.x-t.xo,u=n.x-n.xo,a=Math.abs(s),i=Math.abs(o),c=a+i,0===c?void(e.x=e.xo+(l+u)/2):void(e.x=e.xo+(l*i+u*a)/c);s=r.distance(e,t,!0,!0),o=r.distance(e,n,!0,!0),l=r.distance(t,t,!1,!0),u=r.distance(n,n,!1,!0),a=Math.abs(s),i=Math.abs(o),c=a+i,0!==c?yr.setRelative(e,e,(l*i+u*a)/c,r,!0):yr.setRelative(e,e,(l+u)/2,r,!0)},normalSlope:Number.NEGATIVE_INFINITY,setRelative:function(e,t,n,r,s){if(!r||r===this)return void(e.x=(s?t.xo:t.x)+n);const o=s?t.xo:t.x,a=s?t.yo:t.y,i=o+n*r.x,l=a+n*r.y;e.x=i+(e.y-l)/r.normalSlope},slope:0,touch:function(e){e.xTouched=!0},touched:function(e){return e.xTouched},untouch:function(e){e.xTouched=!1}},vr={x:0,y:1,axis:"y",distance:function(e,t,n,r){return(n?e.yo:e.y)-(r?t.yo:t.y)},interpolate:function(e,t,n,r){let s,o,a,i,l,u,c;if(!r||r===this)return s=e.yo-t.yo,o=e.yo-n.yo,l=t.y-t.yo,u=n.y-n.yo,a=Math.abs(s),i=Math.abs(o),c=a+i,0===c?void(e.y=e.yo+(l+u)/2):void(e.y=e.yo+(l*i+u*a)/c);s=r.distance(e,t,!0,!0),o=r.distance(e,n,!0,!0),l=r.distance(t,t,!1,!0),u=r.distance(n,n,!1,!0),a=Math.abs(s),i=Math.abs(o),c=a+i,0!==c?vr.setRelative(e,e,(l*i+u*a)/c,r,!0):vr.setRelative(e,e,(l+u)/2,r,!0)},normalSlope:0,setRelative:function(e,t,n,r,s){if(!r||r===this)return void(e.y=(s?t.yo:t.y)+n);const o=s?t.xo:t.x,a=s?t.yo:t.y,i=o+n*r.x,l=a+n*r.y;e.y=l+r.normalSlope*(e.x-i)},slope:Number.POSITIVE_INFINITY,touch:function(e){e.yTouched=!0},touched:function(e){return e.yTouched},untouch:function(e){e.yTouched=!1}};function br(e,t){this.x=e,this.y=t,this.axis=void 0,this.slope=t/e,this.normalSlope=-e/t,Object.freeze(this)}function Sr(e,t){const n=Math.sqrt(e*e+t*t);return t/=n,1===(e/=n)&&0===t?yr:0===e&&1===t?vr:new br(e,t)}function xr(e,t,n,r){this.x=this.xo=Math.round(64*e)/64,this.y=this.yo=Math.round(64*t)/64,this.lastPointOfContour=n,this.onCurve=r,this.prevPointOnContour=void 0,this.nextPointOnContour=void 0,this.xTouched=!1,this.yTouched=!1,Object.preventExtensions(this)}Object.freeze(yr),Object.freeze(vr),br.prototype.distance=function(e,t,n,r){return this.x*yr.distance(e,t,n,r)+this.y*vr.distance(e,t,n,r)},br.prototype.interpolate=function(e,t,n,r){let s,o,a,i,l,u,c;a=r.distance(e,t,!0,!0),i=r.distance(e,n,!0,!0),s=r.distance(t,t,!1,!0),o=r.distance(n,n,!1,!0),l=Math.abs(a),u=Math.abs(i),c=l+u,0!==c?this.setRelative(e,e,(s*u+o*l)/c,r,!0):this.setRelative(e,e,(s+o)/2,r,!0)},br.prototype.setRelative=function(e,t,n,r,s){r=r||this;const o=s?t.xo:t.x,a=s?t.yo:t.y,i=o+n*r.x,l=a+n*r.y,u=r.normalSlope,c=this.slope,p=e.x,h=e.y;e.x=(c*p-u*i+l-h)/(c-u),e.y=c*(e.x-p)+h},br.prototype.touch=function(e){e.xTouched=!0,e.yTouched=!0},xr.prototype.nextTouched=function(e){let t=this.nextPointOnContour;for(;!e.touched(t)&&t!==this;)t=t.nextPointOnContour;return t},xr.prototype.prevTouched=function(e){let t=this.prevPointOnContour;for(;!e.touched(t)&&t!==this;)t=t.prevPointOnContour;return t};var Tr=Object.freeze(new xr(0,0)),kr={cvCutIn:17/16,deltaBase:9,deltaShift:.125,loop:1,minDis:1,autoFlip:!0};function Or(e,t){switch(this.env=e,this.stack=[],this.prog=t,e){case"glyf":this.zp0=this.zp1=this.zp2=1,this.rp0=this.rp1=this.rp2=0;case"prep":this.fv=this.pv=this.dpv=yr,this.round=pr}}function Ur(e){const t=e.tZone=new Array(e.gZone.length);for(let e=0;e<t.length;e++)t[e]=new xr(0,0)}function Rr(e,t){const n=e.prog;let r,s=e.ip,o=1;do{if(r=n[++s],88===r)o++;else if(89===r)o--;else if(64===r)s+=n[s+1]+1;else if(65===r)s+=2*n[s+1]+1;else if(r>=176&&r<=183)s+=r-176+1;else if(r>=184&&r<=191)s+=2*(r-184+1);else if(t&&1===o&&27===r)break}while(o>0);e.ip=s}function wr(e,t){t.fv=t.pv=t.dpv=e}function Ir(e,t){t.pv=t.dpv=e}function Cr(e,t){t.fv=e}function Lr(e,t){const n=t.stack,r=n.pop(),s=n.pop(),o=t.z2[r],a=t.z1[s];let i,l;e?(i=o.y-a.y,l=a.x-o.x):(i=a.x-o.x,l=a.y-o.y),t.pv=t.dpv=Sr(i,l)}function Er(e,t){const n=t.stack,r=n.pop(),s=n.pop(),o=t.z2[r],a=t.z1[s];let i,l;e?(i=o.y-a.y,l=a.x-o.x):(i=a.x-o.x,l=a.y-o.y),t.fv=Sr(i,l)}function Dr(e){e.stack.pop()}function Mr(e,t){const n=t.stack.pop(),r=t.z0[n],s=t.fv,o=t.pv;let a=o.distance(r,Tr);e&&(a=t.round(a)),s.setRelative(r,Tr,a,o),s.touch(r),t.rp0=t.rp1=n}function Ar(e,t){const n=t.z2,r=n.length-2;let s,o,a;for(let t=0;t<r;t++)s=n[t],e.touched(s)||(o=s.prevTouched(e),o!==s&&(a=s.nextTouched(e),o===a&&e.setRelative(s,s,e.distance(o,o,!1,!0),e,!0),e.interpolate(s,o,a,e)))}function Fr(e,t){const n=t.stack,r=e?t.rp1:t.rp2,s=(e?t.z0:t.z1)[r],o=t.fv,a=t.pv;let i=t.loop;const l=t.z2;for(;i--;){const e=n.pop(),t=l[e],r=a.distance(s,s,!1,!0);o.setRelative(t,t,r,a),o.touch(t)}t.loop=1}function Pr(e,t){const n=t.stack,r=e?t.rp1:t.rp2,s=(e?t.z0:t.z1)[r],o=t.fv,a=t.pv,i=n.pop(),l=t.z2[t.contours[i]];let u=l;const c=a.distance(s,s,!1,!0);do{u!==s&&o.setRelative(u,u,c,a),u=u.nextPointOnContour}while(u!==l)}function Nr(e,t){const n=t.stack,r=e?t.rp1:t.rp2,s=(e?t.z0:t.z1)[r],o=t.fv,a=t.pv,i=n.pop();let l,u;switch(i){case 0:l=t.tZone;break;case 1:l=t.gZone;break;default:throw new Error("Invalid zone")}const c=a.distance(s,s,!1,!0),p=l.length-2;for(let e=0;e<p;e++)u=l[e],o.setRelative(u,u,c,a)}function Gr(e,t){const n=t.stack,r=n.pop()/64,s=n.pop(),o=t.z1[s],a=t.z0[t.rp0],i=t.fv,l=t.pv;i.setRelative(o,a,r,l),i.touch(o),t.rp1=t.rp0,t.rp2=s,e&&(t.rp0=s)}function Br(e,t){const n=t.stack,r=n.pop(),s=n.pop(),o=t.z0[s],a=t.fv,i=t.pv;let l=t.cvt[r];let u=i.distance(o,Tr);e&&(Math.abs(u-l)<t.cvCutIn&&(u=l),u=t.round(u)),a.setRelative(o,Tr,u,i),0===t.zp0&&(o.xo=o.x,o.yo=o.y),a.touch(o),t.rp0=t.rp1=s}function Hr(e,t){const n=t.stack,r=n.pop(),s=t.z2[r];n.push(64*t.dpv.distance(s,Tr,e,!1))}function zr(e,t){const n=t.stack,r=n.pop(),s=n.pop(),o=t.z1[r],a=t.z0[s],i=t.dpv.distance(a,o,e,e);t.stack.push(Math.round(64*i))}function Vr(e,t){const n=t.stack,r=n.pop(),s=t.fv,o=t.pv,a=t.ppem,i=t.deltaBase+16*(e-1),l=t.deltaShift,u=t.z0;for(let e=0;e<r;e++){const e=n.pop(),t=n.pop();if(i+((240&t)>>4)!==a)continue;let r=(15&t)-8;r>=0&&r++;const c=u[e];s.setRelative(c,c,r*l,o)}}function qr(e,t){const n=t.stack,r=n.pop();n.push(64*t.round(r/64))}function _r(e,t){const n=t.stack,r=n.pop(),s=t.ppem,o=t.deltaBase+16*(e-1),a=t.deltaShift;for(let e=0;e<r;e++){const e=n.pop(),r=n.pop();if(o+((240&r)>>4)!==s)continue;let i=(15&r)-8;i>=0&&i++;const l=i*a;0,t.cvt[e]+=l}}function jr(e,t){const n=t.stack,r=n.pop(),s=n.pop(),o=t.z2[r],a=t.z1[s];let i,l;e?(i=o.y-a.y,l=a.x-o.x):(i=a.x-o.x,l=a.y-o.y),t.dpv=Sr(i,l)}function Wr(e,t){const n=t.stack,r=t.prog;let s=t.ip;for(let t=0;t<e;t++)n.push(r[++s]);t.ip=s}function $r(e,t){let n=t.ip;const r=t.prog,s=t.stack;for(let t=0;t<e;t++){let e=r[++n]<<8|r[++n];32768&e&&(e=-(1+(65535^e))),s.push(e)}t.ip=n}function Yr(e,t,n,r,s,o){const a=o.stack,i=e&&a.pop(),l=a.pop(),u=o.rp0,c=o.z0[u],p=o.z1[l],h=o.minDis,f=o.fv,d=o.dpv;let g,m,y,v;m=g=d.distance(p,c,!0,!0),y=m>=0?1:-1,m=Math.abs(m),e&&(v=o.cvt[i],r&&Math.abs(m-v)<o.cvCutIn&&(m=v)),n&&m<h&&(m=h),r&&(m=o.round(m)),f.setRelative(p,c,y*m,d),f.touch(p),o.rp1=o.rp0,o.rp2=l,t&&(o.rp0=l)}ur.prototype.exec=function(e,t){if("number"!=typeof t)throw new Error("Point size is not a number!");if(this._errorState>2)return;const n=this.font;let r=this._prepState;if(!r||r.ppem!==t){let e=this._fpgmState;if(!e){Or.prototype=kr,e=this._fpgmState=new Or("fpgm",n.tables.fpgm),e.funcs=[],e.font=n;try{rr(e)}catch(e){return console.log("Hinting error in FPGM:"+e),void(this._errorState=3)}}Or.prototype=e,r=this._prepState=new Or("prep",n.tables.prep),r.ppem=t;const s=n.variation&&n.variation.process.getCvarTransform()||n.tables.cvt;if(s){const e=r.cvt=new Array(s.length),o=t/n.unitsPerEm;for(let t=0;t<s.length;t++)e[t]=s[t]*o}else r.cvt=[];0;try{rr(r)}catch(e){this._errorState<2&&console.log("Hinting error in PREP:"+e),this._errorState=2}}if(!(this._errorState>1))try{return sr(e,r)}catch(e){return this._errorState<1&&(console.log("Hinting error:"+e),console.log("Note: further hinting errors are silenced")),void(this._errorState=1)}},sr=function(e,t){const n=t.ppem/t.font.unitsPerEm,r=n;let s,o,a,i=e.components;if(Or.prototype=t,i){const l=t.font;o=[],s=[];for(let e=0;e<i.length;e++){const t=i[e],u=l.glyphs.get(t.glyphIndex);a=new Or("glyf",u.instructions),or(u,a,n,r);const c=Math.round(t.dx*n),p=Math.round(t.dy*r),h=a.gZone,f=a.contours;for(let e=0;e<h.length;e++){const t=h[e];t.xTouched=t.yTouched=!1,t.xo=t.x=t.x+c,t.yo=t.y=t.y+p}const d=o.length;o.push.apply(o,h);for(let e=0;e<f.length;e++)s.push(f[e]+d)}e.instructions&&!a.inhibitGridFit&&(a=new Or("glyf",e.instructions),a.gZone=a.z0=a.z1=a.z2=o,a.contours=s,o.push(new xr(0,0),new xr(Math.round(e.advanceWidth*n),0)),rr(a),o.length-=2)}else a=new Or("glyf",e.instructions),or(e,a,n,r),o=a.gZone;return o},or=function(e,t,n,r){const s=e.points||[],o=s.length,a=t.gZone=t.z0=t.z1=t.z2=[],i=t.contours=[];let l,u,c;for(let e=0;e<o;e++)l=s[e],a[e]=new xr(l.x*n,l.y*r,l.lastPointOfContour,l.onCurve);for(let e=0;e<o;e++)l=a[e],u||(u=l,i.push(e)),l.lastPointOfContour?(l.nextPointOnContour=u,u.prevPointOnContour=l,u=void 0):(c=a[e+1],l.nextPointOnContour=c,c.prevPointOnContour=l);t.inhibitGridFit||(a.push(new xr(0,0),new xr(Math.round(e.advanceWidth*n),0)),rr(t),a.length-=2)},rr=function(e){let t=e.prog;if(!t)return;const n=t.length;let r;for(e.ip=0;e.ip<n;e.ip++){if(r=nr[t[e.ip]],!r)throw new Error("unknown instruction: 0x"+Number(t[e.ip]).toString(16));r(e)}},nr=[wr.bind(void 0,vr),wr.bind(void 0,yr),Ir.bind(void 0,vr),Ir.bind(void 0,yr),Cr.bind(void 0,vr),Cr.bind(void 0,yr),Lr.bind(void 0,0),Lr.bind(void 0,1),Er.bind(void 0,0),Er.bind(void 0,1),function(e){const t=e.stack,n=t.pop(),r=t.pop();e.pv=e.dpv=Sr(r,n)},function(e){const t=e.stack,n=t.pop(),r=t.pop();e.fv=Sr(r,n)},function(e){const t=e.stack,n=e.pv;t.push(16384*n.x),t.push(16384*n.y)},function(e){const t=e.stack,n=e.fv;t.push(16384*n.x),t.push(16384*n.y)},function(e){e.fv=e.pv},function(e){const t=e.stack,n=t.pop(),r=t.pop(),s=t.pop(),o=t.pop(),a=t.pop(),i=e.z0,l=e.z1,u=i[n],c=i[r],p=l[s],h=l[o],f=e.z2[a],d=u.x,g=u.y,m=c.x,y=c.y,v=p.x,b=p.y,S=h.x,x=h.y,T=(d-m)*(b-x)-(g-y)*(v-S),k=d*y-g*m,O=v*x-b*S;f.x=(k*(v-S)-O*(d-m))/T,f.y=(k*(b-x)-O*(g-y))/T},function(e){e.rp0=e.stack.pop()},function(e){e.rp1=e.stack.pop()},function(e){e.rp2=e.stack.pop()},function(e){const t=e.stack.pop();switch(e.zp0=t,t){case 0:e.tZone||Ur(e),e.z0=e.tZone;break;case 1:e.z0=e.gZone;break;default:throw new Error("Invalid zone pointer")}},function(e){const t=e.stack.pop();switch(e.zp1=t,t){case 0:e.tZone||Ur(e),e.z1=e.tZone;break;case 1:e.z1=e.gZone;break;default:throw new Error("Invalid zone pointer")}},function(e){const t=e.stack.pop();switch(e.zp2=t,t){case 0:e.tZone||Ur(e),e.z2=e.tZone;break;case 1:e.z2=e.gZone;break;default:throw new Error("Invalid zone pointer")}},function(e){const t=e.stack.pop();switch(e.zp0=e.zp1=e.zp2=t,t){case 0:e.tZone||Ur(e),e.z0=e.z1=e.z2=e.tZone;break;case 1:e.z0=e.z1=e.z2=e.gZone;break;default:throw new Error("Invalid zone pointer")}},function(e){e.loop=e.stack.pop()},function(e){e.round=pr},function(e){e.round=fr},function(e){const t=e.stack.pop();e.minDis=t/64},function(e){Rr(e,!1)},function(e){const t=e.stack.pop();e.ip+=t-1},function(e){const t=e.stack.pop();e.cvCutIn=t/64},void 0,void 0,function(e){const t=e.stack;t.push(t[t.length-1])},Dr,function(e){e.stack.length=0},function(e){const t=e.stack,n=t.pop(),r=t.pop();t.push(n),t.push(r)},function(e){const t=e.stack;t.push(t.length)},function(e){const t=e.stack,n=t.pop();t.push(t[t.length-n])},function(e){const t=e.stack,n=t.pop();t.push(t.splice(t.length-n,1)[0])},void 0,void 0,void 0,function(e){const t=e.stack,n=t.pop(),r=t.pop(),s=e.ip,o=e.prog;e.prog=e.funcs[n];for(let t=0;t<r;t++)rr(e);e.ip=s,e.prog=o},function(e){const t=e.stack.pop(),n=e.ip,r=e.prog;e.prog=e.funcs[t],rr(e),e.ip=n,e.prog=r},function(e){if("fpgm"!==e.env)throw new Error("FDEF not allowed here");const t=e.stack,n=e.prog;let r=e.ip;const s=t.pop(),o=r;for(0;45!==n[++r];);e.ip=r,e.funcs[s]=n.slice(o+1,r)},void 0,Mr.bind(void 0,0),Mr.bind(void 0,1),Ar.bind(void 0,vr),Ar.bind(void 0,yr),Fr.bind(void 0,0),Fr.bind(void 0,1),Pr.bind(void 0,0),Pr.bind(void 0,1),Nr.bind(void 0,0),Nr.bind(void 0,1),function(e){const t=e.stack;let n=e.loop;const r=e.fv,s=t.pop()/64,o=e.z2;for(;n--;){const e=t.pop(),n=o[e];0,r.setRelative(n,n,s),r.touch(n)}e.loop=1},function(e){const t=e.stack,n=e.rp1,r=e.rp2;let s=e.loop;const o=e.z0[n],a=e.z1[r],i=e.fv,l=e.dpv,u=e.z2;for(;s--;){const e=t.pop(),n=u[e];0,i.interpolate(n,o,a,l),i.touch(n)}e.loop=1},Gr.bind(void 0,0),Gr.bind(void 0,1),function(e){const t=e.stack,n=e.rp0,r=e.z0[n];let s=e.loop;const o=e.fv,a=e.pv,i=e.z1;for(;s--;){const e=t.pop(),n=i[e];0,o.setRelative(n,r,0,a),o.touch(n)}e.loop=1},function(e){e.round=hr},Br.bind(void 0,0),Br.bind(void 0,1),function(e){const t=e.prog;let n=e.ip;const r=e.stack,s=t[++n];for(let e=0;e<s;e++)r.push(t[++n]);e.ip=n},function(e){let t=e.ip;const n=e.prog,r=e.stack,s=n[++t];for(let e=0;e<s;e++){let e=n[++t]<<8|n[++t];32768&e&&(e=-(1+(65535^e))),r.push(e)}e.ip=t},function(e){const t=e.stack;let n=e.store;n||(n=e.store=[]);const r=t.pop(),s=t.pop();n[s]=r},function(e){const t=e.stack,n=e.store,r=t.pop(),s=n&&n[r]||0;t.push(s)},function(e){const t=e.stack,n=t.pop(),r=t.pop();e.cvt[r]=n/64},function(e){const t=e.stack,n=t.pop();t.push(64*e.cvt[n])},Hr.bind(void 0,0),Hr.bind(void 0,1),void 0,zr.bind(void 0,0),zr.bind(void 0,1),function(e){e.stack.push(e.ppem)},void 0,function(e){e.autoFlip=!0},void 0,void 0,function(e){const t=e.stack,n=t.pop(),r=t.pop();t.push(r<n?1:0)},function(e){const t=e.stack,n=t.pop(),r=t.pop();t.push(r<=n?1:0)},function(e){const t=e.stack,n=t.pop(),r=t.pop();t.push(r>n?1:0)},function(e){const t=e.stack,n=t.pop(),r=t.pop();t.push(r>=n?1:0)},function(e){const t=e.stack,n=t.pop(),r=t.pop();t.push(n===r?1:0)},function(e){const t=e.stack,n=t.pop(),r=t.pop();t.push(n!==r?1:0)},function(e){const t=e.stack,n=t.pop();t.push(1&Math.trunc(n)?1:0)},function(e){const t=e.stack,n=t.pop();t.push(1&Math.trunc(n)?0:1)},function(e){let t=e.stack.pop();t||Rr(e,!0)},function(e){0},function(e){const t=e.stack,n=t.pop(),r=t.pop();t.push(n&&r?1:0)},function(e){const t=e.stack,n=t.pop(),r=t.pop();t.push(n||r?1:0)},function(e){const t=e.stack,n=t.pop();t.push(n?0:1)},Vr.bind(void 0,1),function(e){const t=e.stack.pop();e.deltaBase=t},function(e){const t=e.stack.pop();e.deltaShift=Math.pow(.5,t)},function(e){const t=e.stack,n=t.pop(),r=t.pop();t.push(r+n)},function(e){const t=e.stack,n=t.pop(),r=t.pop();t.push(r-n)},function(e){const t=e.stack,n=t.pop(),r=t.pop();t.push(64*r/n)},function(e){const t=e.stack,n=t.pop(),r=t.pop();t.push(r*n/64)},function(e){const t=e.stack,n=t.pop();t.push(Math.abs(n))},function(e){const t=e.stack;let n=t.pop();t.push(-n)},function(e){const t=e.stack,n=t.pop();t.push(64*Math.floor(n/64))},function(e){const t=e.stack,n=t.pop();t.push(64*Math.ceil(n/64))},qr.bind(void 0,0),qr.bind(void 0,1),qr.bind(void 0,2),qr.bind(void 0,3),void 0,void 0,void 0,void 0,function(e){const t=e.stack,n=t.pop(),r=t.pop();e.cvt[r]=n*e.ppem/e.font.unitsPerEm},Vr.bind(void 0,2),Vr.bind(void 0,3),_r.bind(void 0,1),_r.bind(void 0,2),_r.bind(void 0,3),function(e){let t,n=e.stack.pop();switch(e.round=mr,192&n){case 0:t=.5;break;case 64:t=1;break;case 128:t=2;break;default:throw new Error("invalid SROUND value")}switch(e.srPeriod=t,48&n){case 0:e.srPhase=0;break;case 16:e.srPhase=.25*t;break;case 32:e.srPhase=.5*t;break;case 48:e.srPhase=.75*t;break;default:throw new Error("invalid SROUND value")}n&=15,e.srThreshold=0===n?0:(n/8-.5)*t},function(e){let t,n=e.stack.pop();switch(e.round=mr,192&n){case 0:t=Math.sqrt(2)/2;break;case 64:t=Math.sqrt(2);break;case 128:t=2*Math.sqrt(2);break;default:throw new Error("invalid S45ROUND value")}switch(e.srPeriod=t,48&n){case 0:e.srPhase=0;break;case 16:e.srPhase=.25*t;break;case 32:e.srPhase=.5*t;break;case 48:e.srPhase=.75*t;break;default:throw new Error("invalid S45ROUND value")}n&=15,e.srThreshold=0===n?0:(n/8-.5)*t},void 0,void 0,function(e){e.round=cr},void 0,function(e){e.round=dr},function(e){e.round=gr},Dr,Dr,void 0,void 0,void 0,void 0,void 0,function(e){e.stack.pop()},jr.bind(void 0,0),jr.bind(void 0,1),function(e){const t=e.stack,n=t.pop();let r=0;1&n&&(r=35),32&n&&(r|=4096),t.push(r)},void 0,function(e){const t=e.stack,n=t.pop(),r=t.pop(),s=t.pop();t.push(r),t.push(n),t.push(s)},function(e){const t=e.stack,n=t.pop(),r=t.pop();t.push(Math.max(r,n))},function(e){const t=e.stack,n=t.pop(),r=t.pop();t.push(Math.min(r,n))},function(e){e.stack.pop()},function(e){const t=e.stack.pop();let n=e.stack.pop();switch(t){case 1:return void(e.inhibitGridFit=!!n);case 2:return void(e.ignoreCvt=!!n);default:throw new Error("invalid INSTCTRL[] selector")}},void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,void 0,Wr.bind(void 0,1),Wr.bind(void 0,2),Wr.bind(void 0,3),Wr.bind(void 0,4),Wr.bind(void 0,5),Wr.bind(void 0,6),Wr.bind(void 0,7),Wr.bind(void 0,8),$r.bind(void 0,1),$r.bind(void 0,2),$r.bind(void 0,3),$r.bind(void 0,4),$r.bind(void 0,5),$r.bind(void 0,6),$r.bind(void 0,7),$r.bind(void 0,8),Yr.bind(void 0,0,0,0,0,0),Yr.bind(void 0,0,0,0,0,1),Yr.bind(void 0,0,0,0,0,2),Yr.bind(void 0,0,0,0,0,3),Yr.bind(void 0,0,0,0,1,0),Yr.bind(void 0,0,0,0,1,1),Yr.bind(void 0,0,0,0,1,2),Yr.bind(void 0,0,0,0,1,3),Yr.bind(void 0,0,0,1,0,0),Yr.bind(void 0,0,0,1,0,1),Yr.bind(void 0,0,0,1,0,2),Yr.bind(void 0,0,0,1,0,3),Yr.bind(void 0,0,0,1,1,0),Yr.bind(void 0,0,0,1,1,1),Yr.bind(void 0,0,0,1,1,2),Yr.bind(void 0,0,0,1,1,3),Yr.bind(void 0,0,1,0,0,0),Yr.bind(void 0,0,1,0,0,1),Yr.bind(void 0,0,1,0,0,2),Yr.bind(void 0,0,1,0,0,3),Yr.bind(void 0,0,1,0,1,0),Yr.bind(void 0,0,1,0,1,1),Yr.bind(void 0,0,1,0,1,2),Yr.bind(void 0,0,1,0,1,3),Yr.bind(void 0,0,1,1,0,0),Yr.bind(void 0,0,1,1,0,1),Yr.bind(void 0,0,1,1,0,2),Yr.bind(void 0,0,1,1,0,3),Yr.bind(void 0,0,1,1,1,0),Yr.bind(void 0,0,1,1,1,1),Yr.bind(void 0,0,1,1,1,2),Yr.bind(void 0,0,1,1,1,3),Yr.bind(void 0,1,0,0,0,0),Yr.bind(void 0,1,0,0,0,1),Yr.bind(void 0,1,0,0,0,2),Yr.bind(void 0,1,0,0,0,3),Yr.bind(void 0,1,0,0,1,0),Yr.bind(void 0,1,0,0,1,1),Yr.bind(void 0,1,0,0,1,2),Yr.bind(void 0,1,0,0,1,3),Yr.bind(void 0,1,0,1,0,0),Yr.bind(void 0,1,0,1,0,1),Yr.bind(void 0,1,0,1,0,2),Yr.bind(void 0,1,0,1,0,3),Yr.bind(void 0,1,0,1,1,0),Yr.bind(void 0,1,0,1,1,1),Yr.bind(void 0,1,0,1,1,2),Yr.bind(void 0,1,0,1,1,3),Yr.bind(void 0,1,1,0,0,0),Yr.bind(void 0,1,1,0,0,1),Yr.bind(void 0,1,1,0,0,2),Yr.bind(void 0,1,1,0,0,3),Yr.bind(void 0,1,1,0,1,0),Yr.bind(void 0,1,1,0,1,1),Yr.bind(void 0,1,1,0,1,2),Yr.bind(void 0,1,1,0,1,3),Yr.bind(void 0,1,1,1,0,0),Yr.bind(void 0,1,1,1,0,1),Yr.bind(void 0,1,1,1,0,2),Yr.bind(void 0,1,1,1,0,3),Yr.bind(void 0,1,1,1,1,0),Yr.bind(void 0,1,1,1,1,1),Yr.bind(void 0,1,1,1,1,2),Yr.bind(void 0,1,1,1,1,3)];var Xr=ur;function Zr(e){this.char=e,this.state={},this.activeState=null}function Qr(e,t,n){this.contextName=n,this.startIndex=e,this.endOffset=t}function Kr(e,t,n){this.contextName=e,this.openRange=null,this.ranges=[],this.checkStart=t,this.checkEnd=n}function Jr(e,t){this.context=e,this.index=t,this.length=e.length,this.current=e[t],this.backtrack=e.slice(0,t),this.lookahead=e.slice(t+1)}function es(e){this.eventId=e,this.subscribers=[]}function ts(e){const t=["start","end","next","newToken","contextStart","contextEnd","insertToken","removeToken","removeRange","replaceToken","replaceRange","composeRUD","updateContextsRanges"];for(let e=0;e<t.length;e++){const n=t[e];Object.defineProperty(this.events,n,{value:new es(n)})}if(e)for(let n=0;n<t.length;n++){const r=t[n],s=e[r];"function"==typeof s&&this.events[r].subscribe(s)}const n=["insertToken","removeToken","removeRange","replaceToken","replaceRange","composeRUD"];for(let e=0;e<n.length;e++){const t=n[e];this.events[t].subscribe(this.updateContextsRanges)}}function ns(e){this.tokens=[],this.registeredContexts={},this.contextCheckers=[],this.events={},this.registeredModifiers=[],ts.call(this,e)}Zr.prototype.setState=function(e,t){return this.state[e]=t,this.activeState={key:e,value:this.state[e]},this.activeState},Zr.prototype.getState=function(e){return this.state[e]||null},ns.prototype.inboundIndex=function(e){return e>=0&&e<this.tokens.length},ns.prototype.composeRUD=function(e){const t=e.map((e=>this[e[0]].apply(this,e.slice(1).concat(true)))),n=e=>"object"==typeof e&&Object.prototype.hasOwnProperty.call(e,"FAIL");if(t.every(n))return{FAIL:"composeRUD: one or more operations hasn't completed successfully",report:t.filter(n)};this.dispatch("composeRUD",[t.filter((e=>!n(e)))])},ns.prototype.replaceRange=function(e,t,n,r){t=null!==t?t:this.tokens.length;const s=n.every((e=>e instanceof Zr));if(!isNaN(e)&&this.inboundIndex(e)&&s){const s=this.tokens.splice.apply(this.tokens,[e,t].concat(n));return r||this.dispatch("replaceToken",[e,t,n]),[s,n]}return{FAIL:"replaceRange: invalid tokens or startIndex."}},ns.prototype.replaceToken=function(e,t,n){if(!isNaN(e)&&this.inboundIndex(e)&&t instanceof Zr){const r=this.tokens.splice(e,1,t);return n||this.dispatch("replaceToken",[e,t]),[r[0],t]}return{FAIL:"replaceToken: invalid token or index."}},ns.prototype.removeRange=function(e,t,n){t=isNaN(t)?this.tokens.length:t;const r=this.tokens.splice(e,t);return n||this.dispatch("removeRange",[r,e,t]),r},ns.prototype.removeToken=function(e,t){if(!isNaN(e)&&this.inboundIndex(e)){const n=this.tokens.splice(e,1);return t||this.dispatch("removeToken",[n,e]),n}return{FAIL:"removeToken: invalid token index."}},ns.prototype.insertToken=function(e,t,n){return e.every((e=>e instanceof Zr))?(this.tokens.splice.apply(this.tokens,[t,0].concat(e)),n||this.dispatch("insertToken",[e,t]),e):{FAIL:"insertToken: invalid token(s)."}},ns.prototype.registerModifier=function(e,t,n){this.events.newToken.subscribe((function(r,s){const o=[r,s],a=[r,s];if(null===t||!0===t.apply(this,o)){let t=n.apply(this,a);r.setState(e,t)}})),this.registeredModifiers.push(e)},es.prototype.subscribe=function(e){return"function"==typeof e?this.subscribers.push(e)-1:{FAIL:`invalid '${this.eventId}' event handler`}},es.prototype.unsubscribe=function(e){this.subscribers.splice(e,1)},Jr.prototype.setCurrentIndex=function(e){this.index=e,this.current=this.context[e],this.backtrack=this.context.slice(0,e),this.lookahead=this.context.slice(e+1)},Jr.prototype.get=function(e){switch(!0){case 0===e:return this.current;case e<0&&Math.abs(e)<=this.backtrack.length:return this.backtrack.slice(e)[0];case e>0&&e<=this.lookahead.length:return this.lookahead[e-1];default:return null}},ns.prototype.rangeToText=function(e){if(e instanceof Qr)return this.getRangeTokens(e).map((e=>e.char)).join("")},ns.prototype.getText=function(){return this.tokens.map((e=>e.char)).join("")},ns.prototype.getContext=function(e){let t=this.registeredContexts[e];return t||null},ns.prototype.on=function(e,t){const n=this.events[e];return n?n.subscribe(t):null},ns.prototype.dispatch=function(e,t){const n=this.events[e];if(n instanceof es)for(let e=0;e<n.subscribers.length;e++){n.subscribers[e].apply(this,t||[])}},ns.prototype.registerContextChecker=function(e,t,n){if(this.getContext(e))return{FAIL:`context name '${e}' is already registered.`};if("function"!=typeof t)return{FAIL:"missing context start check."};if("function"!=typeof n)return{FAIL:"missing context end check."};const r=new Kr(e,t,n);return this.registeredContexts[e]=r,this.contextCheckers.push(r),r},ns.prototype.getRangeTokens=function(e){const t=e.startIndex+e.endOffset;return[].concat(this.tokens.slice(e.startIndex,t))},ns.prototype.getContextRanges=function(e){const t=this.getContext(e);return t?t.ranges:{FAIL:`context checker '${e}' is not registered.`}},ns.prototype.resetContextsRanges=function(){const e=this.registeredContexts;for(const t in e)if(Object.prototype.hasOwnProperty.call(e,t)){e[t].ranges=[]}},ns.prototype.updateContextsRanges=function(){this.resetContextsRanges();const e=this.tokens.map((e=>e.char));for(let t=0;t<e.length;t++){const n=new Jr(e,t);this.runContextCheck(n)}this.dispatch("updateContextsRanges",[this.registeredContexts])},ns.prototype.setEndOffset=function(e,t){let n=new Qr(this.getContext(t).openRange.startIndex,e,t);const r=this.getContext(t).ranges;return n.rangeId=`${t}.${r.length}`,r.push(n),this.getContext(t).openRange=null,n},ns.prototype.runContextCheck=function(e){const t=e.index;for(let n=0;n<this.contextCheckers.length;n++){const r=this.contextCheckers[n];let s=r.contextName,o=this.getContext(s).openRange;if(!o&&r.checkStart(e)&&(o=new Qr(t,null,s),this.getContext(s).openRange=o,this.dispatch("contextStart",[s,t])),o&&r.checkEnd(e)){const e=t-o.startIndex+1,n=this.setEndOffset(e,s);this.dispatch("contextEnd",[s,n])}}},ns.prototype.tokenize=function(e){this.tokens=[],this.resetContextsRanges();let t=Array.from(e);this.dispatch("start");for(let e=0;e<t.length;e++){const n=t[e],r=new Jr(t,e);this.dispatch("next",[r]),this.runContextCheck(r);let s=new Zr(n);this.tokens.push(s),this.dispatch("newToken",[s,r])}return this.dispatch("end",[this.tokens]),this.tokens};var rs=ns;function ss(e){return/[\u0600-\u065F\u066A-\u06D2\u06FA-\u06FF]/.test(e)}function os(e){return/[\u0630\u0690\u0621\u0631\u0661\u0671\u0622\u0632\u0672\u0692\u06C2\u0623\u0673\u0693\u06C3\u0624\u0694\u06C4\u0625\u0675\u0695\u06C5\u06E5\u0676\u0696\u06C6\u0627\u0677\u0697\u06C7\u0648\u0688\u0698\u06C8\u0689\u0699\u06C9\u068A\u06CA\u066B\u068B\u06CB\u068C\u068D\u06CD\u06FD\u068E\u06EE\u06FE\u062F\u068F\u06CF\u06EF]/.test(e)}function as(e){return/[\u0600-\u0605\u060C-\u060E\u0610-\u061B\u061E\u064B-\u065F\u0670\u06D6-\u06DC\u06DF-\u06E4\u06E7\u06E8\u06EA-\u06ED]/.test(e)}function is(e){return/[\u0E00-\u0E7F]/.test(e)}function ls(e){return/[A-z]/.test(e)}function us(e){this.font=e,this.features={}}function cs(e){this.id=e.id,this.tag=e.tag,this.substitution=e.substitution}function ps(e,t){if(!e)return-1;switch(t.format){case 1:return t.glyphs.indexOf(e);case 2:{let n=t.ranges;for(let t=0;t<n.length;t++){const r=n[t];if(e>=r.start&&e<=r.end){let t=e-r.start;return r.index+t}}break}default:return-1}return-1}function hs(e,t){return-1===ps(e,t.coverage)?null:e+t.deltaGlyphId}function fs(e,t){let n=ps(e,t.coverage);return-1===n?null:t.substitute[n]}function ds(e,t){let n=[];for(let r=0;r<e.length;r++){const s=e[r];let o=t.current;o=Array.isArray(o)?o[0]:o;const a=ps(o,s);-1!==a&&n.push(a)}return n.length!==e.length?-1:n}function gs(e,t){const n=t.inputCoverage.length+t.lookaheadCoverage.length+t.backtrackCoverage.length;if(e.context.length<n)return[];let r=ds(t.inputCoverage,e);if(-1===r)return[];const s=t.inputCoverage.length-1;if(e.lookahead.length<t.lookaheadCoverage.length)return[];let o=e.lookahead.slice(s);for(;o.length&&as(o[0].char);)o.shift();const a=new Jr(o,0);let i=ds(t.lookaheadCoverage,a),l=[].concat(e.backtrack);for(l.reverse();l.length&&as(l[0].char);)l.shift();if(l.length<t.backtrackCoverage.length)return[];const u=new Jr(l,0);let c=ds(t.backtrackCoverage,u);let p=[];if(r.length===t.inputCoverage.length&&i.length===t.lookaheadCoverage.length&&c.length===t.backtrackCoverage.length)for(let n=0;n<t.lookupRecords.length;n++){const s=t.lookupRecords[n].lookupListIndex,o=this.getLookupByIndex(s);for(let t=0;t<o.subtables.length;t++){let n,s=o.subtables[t],a=this.getSubstitutionType(o,s);if("71"===a?(a=this.getSubstitutionType(s,s.extension),n=this.getLookupMethod(s,s.extension),s=s.extension):n=this.getLookupMethod(o,s),"12"!==a)throw new Error(`Substitution type ${a} is not supported in chaining substitution`);for(let t=0;t<r.length;t++){const r=n(e.get(t));r&&p.push(r)}}}return p}function ms(e,t){let n,r=ps(e.current,t.coverage);if(-1===r)return null;let s=t.ligatureSets[r];for(let t=0;t<s.length;t++){n=s[t];for(let t=0;t<n.components.length;t++){if(e.lookahead[t]!==n.components[t])break;if(t===n.components.length-1)return n}}return null}function ys(e,t){let n=e.current;if(-1===ps(n,t.coverage))return null;for(const r of t.ruleSets)for(const t of r){let r=!0;for(let n=0;n<t.input.length;n++)if(e.lookahead[n]!==t.input[n]){r=!1;break}if(r){let e=[];e.push(n);for(let n=0;n<t.input.length;n++)e.push(t.input[n]);const r=(e,t)=>{const{lookupListIndex:n,sequenceIndex:r}=t,{subtables:s}=this.getLookupByIndex(n);for(const t of s){-1!==ps(e[r],t.coverage)&&(e[r]=t.deltaGlyphId)}};for(let n=0;n<t.lookupRecords.length;n++){r(e,t.lookupRecords[n])}return e}}return null}function vs(e,t){let n=[];for(let r=0;r<t.coverages.length;r++){const s=t.lookupRecords[r],o=t.coverages[r];let a=e.context[e.index+s.sequenceIndex];if(-1===ps(a,o))return null;let i=this.font.tables.gsub.lookups[s.lookupListIndex];for(let e=0;e<i.subtables.length;e++){let t=i.subtables[e],r=ps(a,t.coverage);if(-1===r)return null;switch(i.lookupType){case 1:{let e=t.substitute[r];n.push(e);break}case 2:{let e=t.sequences[r];n.push(e);break}}}}return n}function bs(e,t){let n=ps(e,t.coverage);return-1===n?null:t.sequences[n]}us.prototype.getDefaultScriptFeaturesIndexes=function(){const e=this.font.tables.gsub.scripts;for(let t=0;t<e.length;t++){const n=e[t];if("DFLT"===n.tag)return n.script.defaultLangSys.featureIndexes}return[]},us.prototype.getScriptFeaturesIndexes=function(e){if(!this.font.tables.gsub)return[];if(!e)return this.getDefaultScriptFeaturesIndexes();const t=this.font.tables.gsub.scripts;for(let n=0;n<t.length;n++){const r=t[n];if(r.tag===e&&r.script.defaultLangSys)return r.script.defaultLangSys.featureIndexes;{let t=r.langSysRecords;if(t)for(let n=0;n<t.length;n++){const r=t[n];if(r.tag===e){return r.langSys.featureIndexes}}}}return this.getDefaultScriptFeaturesIndexes()},us.prototype.mapTagsToFeatures=function(e,t){let n={};for(let t=0;t<e.length;t++){const r=e[t].tag,s=e[t].feature;n[r]=s}this.features[t].tags=n},us.prototype.getScriptFeatures=function(e){let t=this.features[e];if(Object.prototype.hasOwnProperty.call(this.features,e))return t;const n=this.getScriptFeaturesIndexes(e);if(!n)return null;const r=this.font.tables.gsub;return t=n.map((e=>r.features[e])),this.features[e]=t,this.mapTagsToFeatures(t,e),t},us.prototype.getSubstitutionType=function(e,t){return e.lookupType.toString()+t.substFormat.toString()},us.prototype.getLookupMethod=function(e,t){let n=this.getSubstitutionType(e,t);switch(n){case"11":return e=>hs.apply(this,[e,t]);case"12":return e=>fs.apply(this,[e,t]);case"63":return e=>gs.apply(this,[e,t]);case"41":return e=>ms.apply(this,[e,t]);case"21":return e=>bs.apply(this,[e,t]);case"51":return e=>ys.apply(this,[e,t]);case"53":return e=>vs.apply(this,[e,t]);default:throw new Error(`substitutionType : ${n} lookupType: ${e.lookupType} - substFormat: ${t.substFormat} is not yet supported`)}},us.prototype.lookupFeature=function(e){let t=e.contextParams,n=t.index;const r=this.getFeature({tag:e.tag,script:e.script});if(!r)return new Error(`font '${(this.font.names.unicode||this.font.names.windows||this.font.names.macintosh).fullName.en}' doesn't support feature '${e.tag}' for script '${e.script}'.`);const s=this.getFeatureLookups(r),o=[].concat(t.context);for(let r=0;r<s.length;r++){const a=s[r],i=this.getLookupSubtables(a);for(let r=0;r<i.length;r++){let s,l,u=i[r],c=this.getSubstitutionType(a,u);switch("71"===c?(c=this.getSubstitutionType(u,u.extension),s=this.getLookupMethod(u,u.extension),u=u.extension):s=this.getLookupMethod(a,u),c){case"11":l=s(t.current),l&&o.splice(n,1,new cs({id:11,tag:e.tag,substitution:l}));break;case"12":l=s(t.current),l&&o.splice(n,1,new cs({id:12,tag:e.tag,substitution:l}));break;case"63":l=s(t),Array.isArray(l)&&l.length&&o.splice(n,1,new cs({id:63,tag:e.tag,substitution:l}));break;case"41":l=s(t),l&&o.splice(n,1,new cs({id:41,tag:e.tag,substitution:l}));break;case"21":l=s(t.current),l&&o.splice(n,1,new cs({id:21,tag:e.tag,substitution:l}));break;case"51":case"53":l=s(t),Array.isArray(l)&&l.length&&o.splice(n,1,new cs({id:parseInt(c),tag:e.tag,substitution:l}))}t=new Jr(o,n),Array.isArray(l)&&!l.length||(l=null)}}return o.length?o:null},us.prototype.supports=function(e){if(!e.script)return!1;this.getScriptFeatures(e.script);const t=Object.prototype.hasOwnProperty.call(this.features,e.script);if(!e.tag)return t;const n=this.features[e.script].some((t=>t.tag===e.tag));return t&&n},us.prototype.getLookupSubtables=function(e){return e.subtables||null},us.prototype.getLookupByIndex=function(e){return this.font.tables.gsub.lookups[e]||null},us.prototype.getFeatureLookups=function(e){return e.lookupListIndexes.map(this.getLookupByIndex.bind(this))},us.prototype.getFeature=function(e){if(!this.font)return{FAIL:"No font was found"};Object.prototype.hasOwnProperty.call(this.features,e.script)||this.getScriptFeatures(e.script);const t=this.features[e.script];return t?t.tags[e.tag]?this.features[e.script].tags[e.tag]:null:{FAIL:`No feature for script ${e.script}`}};var Ss=us;var xs={startCheck:function(e){const t=e.current,n=e.get(-1);return null===n&&ss(t)||!ss(n)&&ss(t)},endCheck:function(e){const t=e.get(1);return null===t||!ss(t)}};var Ts={startCheck:function(e){const t=e.current,n=e.get(-1);return(ss(t)||as(t))&&!ss(n)},endCheck:function(e){const t=e.get(1);switch(!0){case null===t:return!0;case!ss(t)&&!as(t):{const n=/\s/.test(t);if(!n)return!0;if(n){let t=!1;if(t=e.lookahead.some((e=>ss(e)||as(e))),!t)return!0}break}default:return!1}}};function ks(e,t,n){for(let r=0;r<e.substitution.length;r++){const s=e.substitution[r],o=t[n+r];Array.isArray(s)?s.length?o.setState(e.tag,s[0]):o.setState("deleted",!0):o.setState(e.tag,s)}}var Os={11:function(e,t,n){t[n].setState(e.tag,e.substitution)},12:function(e,t,n){t[n].setState(e.tag,e.substitution)},63:ks,41:function(e,t,n){let r=t[n];r.setState(e.tag,e.substitution.ligGlyph);const s=e.substitution.components.length;for(let e=0;e<s;e++)r=t[n+e+1],r.setState("deleted",!0)},51:ks,53:ks};var Us=function(e,t,n){e instanceof cs&&Os[e.id]&&Os[e.id](e,t,n)};function Rs(e){let t=[].concat(e.backtrack);for(let e=t.length-1;e>=0;e--){const n=t[e],r=os(n),s=as(n);if(!r&&!s)return!0;if(r)return!1}return!1}function ws(e){if(os(e.current))return!1;for(let t=0;t<e.lookahead.length;t++){if(!as(e.lookahead[t]))return!0}return!1}var Is=function(e){const t="arab",n=this.featuresTags[t],r=this.tokenizer.getRangeTokens(e);if(1===r.length)return;let s=new Jr(r.map((e=>e.getState("glyphIndex"))),0);const o=new Jr(r.map((e=>e.char)),0);for(let e=0;e<r.length;e++){if(as(r[e].char))continue;s.setCurrentIndex(e),o.setCurrentIndex(e);let a,i=0;switch(Rs(o)&&(i|=1),ws(o)&&(i|=2),i){case 1:a="fina";break;case 2:a="init";break;case 3:a="medi"}if(-1===n.indexOf(a))continue;let l=this.query.lookupFeature({tag:a,script:t,contextParams:s});if(l instanceof Error)console.info(l.message);else for(let e=0;e<l.length;e++){const t=l[e];t instanceof cs&&(Us(t,r,e),s.context[e]=t.substitution)}}};function Cs(e,t){return new Jr(e.map((e=>e.activeState.value)),t||0)}var Ls=function(e){let t=this.tokenizer.getRangeTokens(e),n=Cs(t);for(let e=0;e<n.context.length;e++){n.setCurrentIndex(e);let r=this.query.lookupFeature({tag:"rlig",script:"arab",contextParams:n});if(r.length){for(let n=0;n<r.length;n++){const s=r[n];Us(s,t,e)}n=Cs(t)}}};var Es={startCheck:function(e){return 0===e.index&&e.context.length>1},endCheck:function(e){return e.index===e.context.length-1}};function Ds(e,t){return new Jr(e.map((e=>e.activeState.value)),t||0)}var Ms=function(e){const t="delf",n="ccmp";let r=this.tokenizer.getRangeTokens(e),s=Ds(r);for(let e=0;e<s.context.length;e++){if(!this.query.getFeature({tag:n,script:t,contextParams:s}))continue;s.setCurrentIndex(e);let o=this.query.lookupFeature({tag:n,script:t,contextParams:s});if(o.length){for(let t=0;t<o.length;t++){const n=o[t];Us(n,r,e)}s=Ds(r)}}};var As={startCheck:function(e){const t=e.current,n=e.get(-1);return null===n&&ls(t)||!ls(n)&&ls(t)},endCheck:function(e){const t=e.get(1);return null===t||!ls(t)}};function Fs(e,t){return new Jr(e.map((e=>e.activeState.value)),t||0)}var Ps=function(e){let t=this.tokenizer.getRangeTokens(e),n=Fs(t);for(let e=0;e<n.context.length;e++){n.setCurrentIndex(e);let r=this.query.lookupFeature({tag:"liga",script:"latn",contextParams:n});if(r.length){for(let n=0;n<r.length;n++){const s=r[n];Us(s,t,e)}n=Fs(t)}}};var Ns={startCheck:function(e){const t=e.current,n=e.get(-1);return null===n&&is(t)||!is(n)&&is(t)},endCheck:function(e){const t=e.get(1);return null===t||!is(t)}};function Gs(e,t){return new Jr(e.map((e=>e.activeState.value)),t||0)}var Bs=function(e){let t=this.tokenizer.getRangeTokens(e),n=Gs(t,0);for(let e=0;e<n.context.length;e++){n.setCurrentIndex(e);let r=this.query.lookupFeature({tag:"ccmp",script:"thai",contextParams:n});if(r.length){for(let n=0;n<r.length;n++){const s=r[n];Us(s,t,e)}n=Gs(t,e)}}};function Hs(e,t){return new Jr(e.map((e=>e.activeState.value)),t||0)}var zs=function(e){let t=this.tokenizer.getRangeTokens(e),n=Hs(t,0);for(let e=0;e<n.context.length;e++){n.setCurrentIndex(e);let r=this.query.lookupFeature({tag:"liga",script:"thai",contextParams:n});if(r.length){for(let n=0;n<r.length;n++){const s=r[n];Us(s,t,e)}n=Hs(t,e)}}};function Vs(e,t){return new Jr(e.map((e=>e.activeState.value)),t||0)}var qs=function(e){let t=this.tokenizer.getRangeTokens(e),n=Vs(t,0);for(let e=0;e<n.context.length;e++){n.setCurrentIndex(e);let r=this.query.lookupFeature({tag:"rlig",script:"thai",contextParams:n});if(r.length){for(let n=0;n<r.length;n++){const s=r[n];Us(s,t,e)}n=Vs(t,e)}}};function _s(e){if(null===e)return!1;const t=e.codePointAt(0);return t>=6155&&t<=6157||t>=65024&&t<=65039||t>=917760&&t<=917999}var js={startCheck:function(e){const t=e.current,n=e.get(1);return null===n&&_s(t)||_s(n)},endCheck:function(e){const t=e.get(1);return null===t||!_s(t)}};var Ws=function(e){const t=this.query.font,n=this.tokenizer.getRangeTokens(e);if(n[1].setState("deleted",!0),t.tables.cmap&&t.tables.cmap.varSelectorList){const e=n[0].char.codePointAt(0),r=n[1].char.codePointAt(0),s=t.tables.cmap.varSelectorList[r];if(void 0!==s&&s.nonDefaultUVS){const r=s.nonDefaultUVS.uvsMappings;if(r[e]){const s=r[e].glyphID;void 0!==t.glyphs.glyphs[s]&&n[0].setState("glyphIndex",s)}}}};function $s(e){this.baseDir=e||"ltr",this.tokenizer=new rs,this.featuresTags={}}function Ys(e){const t=this.contextChecks[`${e}Check`];return this.tokenizer.registerContextChecker(e,t.startCheck,t.endCheck)}function Xs(){return Ys.call(this,"ccmpReplacement"),Ys.call(this,"latinWord"),Ys.call(this,"arabicWord"),Ys.call(this,"arabicSentence"),Ys.call(this,"thaiWord"),Ys.call(this,"unicodeVariationSequence"),this.tokenizer.tokenize(this.text)}function Zs(){const e=this.tokenizer.getContextRanges("arabicSentence");for(let t=0;t<e.length;t++){const n=e[t];let r=this.tokenizer.getRangeTokens(n);this.tokenizer.replaceRange(n.startIndex,n.endOffset,r.reverse())}}function Qs(){if(-1===this.tokenizer.registeredModifiers.indexOf("glyphIndex"))throw new Error("glyphIndex modifier is required to apply arabic presentation features.")}function Ks(){if(!Object.prototype.hasOwnProperty.call(this.featuresTags,"arab"))return;Qs.call(this);const e=this.tokenizer.getContextRanges("arabicWord");for(let t=0;t<e.length;t++){const n=e[t];Is.call(this,n)}}function Js(){Qs.call(this);const e=this.tokenizer.getContextRanges("ccmpReplacement");for(let t=0;t<e.length;t++){const n=e[t];Ms.call(this,n)}}function eo(){if(!this.hasFeatureEnabled("arab","rlig"))return;Qs.call(this);const e=this.tokenizer.getContextRanges("arabicWord");for(let t=0;t<e.length;t++){const n=e[t];Ls.call(this,n)}}function to(){if(!this.hasFeatureEnabled("latn","liga"))return;Qs.call(this);const e=this.tokenizer.getContextRanges("latinWord");for(let t=0;t<e.length;t++){const n=e[t];Ps.call(this,n)}}function no(){const e=this.tokenizer.getContextRanges("unicodeVariationSequence");for(let t=0;t<e.length;t++){const n=e[t];Ws.call(this,n)}}function ro(){Qs.call(this);const e=this.tokenizer.getContextRanges("thaiWord");for(let t=0;t<e.length;t++){const n=e[t];this.hasFeatureEnabled("thai","liga")&&zs.call(this,n),this.hasFeatureEnabled("thai","rlig")&&qs.call(this,n),this.hasFeatureEnabled("thai","ccmp")&&Bs.call(this,n)}}$s.prototype.setText=function(e){this.text=e},$s.prototype.contextChecks={ccmpReplacementCheck:Es,latinWordCheck:As,arabicWordCheck:xs,arabicSentenceCheck:Ts,thaiWordCheck:Ns,unicodeVariationSequenceCheck:js},$s.prototype.registerFeatures=function(e,t){const n=t.filter((t=>this.query.supports({script:e,tag:t})));Object.prototype.hasOwnProperty.call(this.featuresTags,e)?this.featuresTags[e]=this.featuresTags[e].concat(n):this.featuresTags[e]=n},$s.prototype.applyFeatures=function(e,t){if(!e)throw new Error("No valid font was provided to apply features");this.query||(this.query=new Ss(e));for(let e=0;e<t.length;e++){const n=t[e];this.query.supports({script:n.script})&&this.registerFeatures(n.script,n.tags)}},$s.prototype.registerModifier=function(e,t,n){this.tokenizer.registerModifier(e,t,n)},$s.prototype.checkContextReady=function(e){return!!this.tokenizer.getContext(e)},$s.prototype.applyFeaturesToContexts=function(){this.checkContextReady("ccmpReplacement")&&Js.call(this),this.checkContextReady("arabicWord")&&(Ks.call(this),eo.call(this)),this.checkContextReady("latinWord")&&to.call(this),this.checkContextReady("arabicSentence")&&Zs.call(this),this.checkContextReady("thaiWord")&&ro.call(this),this.checkContextReady("unicodeVariationSequence")&&no.call(this)},$s.prototype.hasFeatureEnabled=function(e,t){return-1!==(this.featuresTags[e]||[]).indexOf(t)},$s.prototype.processText=function(e){this.text&&this.text===e||(this.setText(e),Xs.call(this),this.applyFeaturesToContexts())},$s.prototype.getBidiText=function(e){return this.processText(e),this.tokenizer.getText()},$s.prototype.getTextGlyphs=function(e){this.processText(e);let t=[];for(let e=0;e<this.tokenizer.tokens.length;e++){const n=this.tokenizer.tokens[e];if(n.state.deleted)continue;const r=n.activeState.value;t.push(Array.isArray(r)?r[0]:r)}return t};var so=$s;function oo(e){return{fontFamily:{en:e.familyName||" "},fontSubfamily:{en:e.styleName||" "},fullName:{en:e.fullName||e.familyName+" "+e.styleName},postScriptName:{en:e.postScriptName||(e.familyName+e.styleName).replace(/\s/g,"")},designer:{en:e.designer||" "},designerURL:{en:e.designerURL||" "},manufacturer:{en:e.manufacturer||" "},manufacturerURL:{en:e.manufacturerURL||" "},license:{en:e.license||" "},licenseURL:{en:e.licenseURL||" "},version:{en:e.version||"Version 0.1"},description:{en:e.description||" "},copyright:{en:e.copyright||" "},trademark:{en:e.trademark||" "}}}function ao(e){if((e=e||{}).tables=e.tables||{},!e.empty){if(!e.familyName)throw"When creating a new Font object, familyName is required.";if(!e.styleName)throw"When creating a new Font object, styleName is required.";if(!e.unitsPerEm)throw"When creating a new Font object, unitsPerEm is required.";if(!e.ascender)throw"When creating a new Font object, ascender is required.";if(e.descender>0)throw"When creating a new Font object, negative descender value is required.";this.names={},this.names.unicode=oo(e),this.names.macintosh=oo(e),this.names.windows=oo(e),this.unitsPerEm=e.unitsPerEm||1e3,this.ascender=e.ascender,this.descender=e.descender,this.createdTimestamp=e.createdTimestamp,this.italicAngle=e.italicAngle||0,this.weightClass=e.weightClass||0;let t=0;e.fsSelection?t=e.fsSelection:(this.italicAngle<0?t|=this.fsSelectionValues.ITALIC:this.italicAngle>0&&(t|=this.fsSelectionValues.OBLIQUE),this.weightClass>=600&&(t|=this.fsSelectionValues.BOLD),0==t&&(t=this.fsSelectionValues.REGULAR)),e.panose&&Array.isArray(e.panose)||(e.panose=[0,0,0,0,0,0,0,0,0]),this.tables=Object.assign(e.tables,{os2:Object.assign({usWeightClass:e.weightClass||this.usWeightClasses.MEDIUM,usWidthClass:e.widthClass||this.usWidthClasses.MEDIUM,bFamilyType:e.panose[0]||0,bSerifStyle:e.panose[1]||0,bWeight:e.panose[2]||0,bProportion:e.panose[3]||0,bContrast:e.panose[4]||0,bStrokeVariation:e.panose[5]||0,bArmStyle:e.panose[6]||0,bLetterform:e.panose[7]||0,bMidline:e.panose[8]||0,bXHeight:e.panose[9]||0,fsSelection:t},e.tables.os2)})}this.supported=!0,this.glyphs=new St.GlyphSet(this,e.glyphs||[]),this.encoding=new rt(this),this.position=new Gn(this),this.substitution=new jn(this),this.tables=this.tables||{},this.tables=new Proxy(this.tables,{set:(e,t,n)=>(e[t]=n,e.fvar&&(e.gvar||e.cff2)&&!this.variation&&(this.variation=new lr(this)),!0)}),this.palettes=new Wn(this),this.layers=new $n(this),this.svgImages=new Yn(this),this._push=null,this._hmtxTableData={},Object.defineProperty(this,"hinting",{get:function(){return this._hinting?this._hinting:"truetype"===this.outlinesFormat?this._hinting=new Xr(this):null}})}ao.prototype.hasChar=function(e){return this.encoding.charToGlyphIndex(e)>0},ao.prototype.charToGlyphIndex=function(e){return this.encoding.charToGlyphIndex(e)},ao.prototype.charToGlyph=function(e){const t=this.charToGlyphIndex(e);let n=this.glyphs.get(t);return n||(n=this.glyphs.get(0)),n},ao.prototype.updateFeatures=function(e){return this.defaultRenderOptions.features.map((t=>"latn"===t.script?{script:"latn",tags:t.tags.filter((t=>e[t]))}:t))},ao.prototype.stringToGlyphIndexes=function(e,t){const n=new so;n.registerModifier("glyphIndex",null,(e=>this.charToGlyphIndex(e.char)));let r=t?this.updateFeatures(t.features):this.defaultRenderOptions.features;return n.applyFeatures(this,r),n.getTextGlyphs(e)},ao.prototype.stringToGlyphs=function(e,t){const n=this.stringToGlyphIndexes(e,t);let r=n.length;const s=new Array(r),o=this.glyphs.get(0);for(let e=0;e<r;e+=1)s[e]=this.glyphs.get(n[e])||o;return s},ao.prototype.nameToGlyphIndex=function(e){return this.glyphNames.nameToGlyphIndex(e)},ao.prototype.nameToGlyph=function(e){const t=this.nameToGlyphIndex(e);let n=this.glyphs.get(t);return n||(n=this.glyphs.get(0)),n},ao.prototype.glyphIndexToName=function(e){return this.glyphNames.glyphIndexToName?this.glyphNames.glyphIndexToName(e):""},ao.prototype.getKerningValue=function(e,t){e=e.index||e,t=t.index||t;const n=this.position.defaultKerningTables;return n?this.position.getKerningValue(n,e,t):this.kerningPairs[e+","+t]||0},ao.prototype.defaultRenderOptions={kerning:!0,features:[{script:"arab",tags:["init","medi","fina","rlig"]},{script:"latn",tags:["liga","rlig"]},{script:"thai",tags:["liga","rlig","ccmp"]}],hinting:!1,usePalette:0,drawLayers:!0,drawSVG:!0},ao.prototype.forEachGlyph=function(e,t,n,r,s,o){t=void 0!==t?t:0,n=void 0!==n?n:0,r=void 0!==r?r:72,s=Object.assign({},this.defaultRenderOptions,s);const a=1/this.unitsPerEm*r,i=this.stringToGlyphs(e,s);let l;if(s.kerning){const e=s.script||this.position.getDefaultScriptName();l=this.position.getKerningTables(e,s.language)}for(let e=0;e<i.length;e+=1){const u=i[e];if(o.call(this,u,t,n,r,s),u.advanceWidth&&(t+=u.advanceWidth*a),s.kerning&&e<i.length-1){t+=(l?this.position.getKerningValue(l,u.index,i[e+1].index):this.getKerningValue(u,i[e+1]))*a}s.letterSpacing?t+=s.letterSpacing*r:s.tracking&&(t+=s.tracking/1e3*r)}return t},ao.prototype.getPath=function(e,t,n,r,s){s=Object.assign({},this.defaultRenderOptions,s);const o=new F;if(o._layers=[],Gt(this,o),o.stroke){const e=1/(o.unitsPerEm||1e3)*r;o.strokeWidth*=e}return this.forEachGlyph(e,t,n,r,s,((e,t,n,r)=>{const a=e.getPath(t,n,r,s,this);if(s.drawSVG||s.drawLayers){const e=a._layers;if(e&&e.length){for(let t=0;t<e.length;t++){const n=e[t];o._layers.push(n)}return}}o.extend(a)})),o},ao.prototype.getPaths=function(e,t,n,r,s){s=Object.assign({},this.defaultRenderOptions,s);const o=[];return this.forEachGlyph(e,t,n,r,s,(function(e,t,n,r){const a=e.getPath(t,n,r,s,this);o.push(a)})),o},ao.prototype.getAdvanceWidth=function(e,t,n){return n=Object.assign({},this.defaultRenderOptions,n),this.forEachGlyph(e,0,0,t,n,(function(){}))},ao.prototype.draw=function(e,t,n,r,s,o){this.getPath(t,n,r,s,o).draw(e)},ao.prototype.drawPoints=function(e,t,n,r,s,o){o=Object.assign({},this.defaultRenderOptions,o),this.forEachGlyph(t,n,r,s,o,(function(t,n,r,s){t.drawPoints(e,n,r,s,o,this)}))},ao.prototype.drawMetrics=function(e,t,n,r,s,o){o=Object.assign({},this.defaultRenderOptions,o),this.forEachGlyph(t,n,r,s,o,(function(t,n,r,s){t.drawMetrics(e,n,r,s)}))},ao.prototype.getEnglishName=function(e){const t=(this.names.unicode||this.names.macintosh||this.names.windows)[e];if(t)return t.en},ao.prototype.validate=function(){const e=[],t=this;function n(t,n){t||(console.warn(`[opentype.js] ${n}`),e.push(n))}function r(e){const r=t.getEnglishName(e);n(r&&r.trim().length>0,"No English "+e+" specified.")}if(r("fontFamily"),r("weightName"),r("manufacturer"),r("copyright"),r("version"),n(this.unitsPerEm>0,"No unitsPerEm specified."),this.tables.colr){const e=this.tables.colr.baseGlyphRecords;let t=-1;for(let r=0;r<e.length;r++){const s=e[r].glyphID;if(n(t<e[r].glyphID,`baseGlyphs must be sorted by GlyphID in ascending order, but glyphID ${s} comes after ${t}`),t>e[r].glyphID)break;t=s}}return e},ao.prototype.toTables=function(){return En.fontToTable(this)},ao.prototype.toBuffer=function(){return console.warn("Font.toBuffer is deprecated. Use Font.toArrayBuffer instead."),this.toArrayBuffer()},ao.prototype.toArrayBuffer=function(){const e=this.toTables().encode(),t=new ArrayBuffer(e.length),n=new Uint8Array(t);for(let t=0;t<e.length;t++)n[t]=e[t];return t},ao.prototype.download=function(){console.error("DEPRECATED: platform-specific actions are to be implemented on user-side")},ao.prototype.fsSelectionValues={ITALIC:1,UNDERSCORE:2,NEGATIVE:4,OUTLINED:8,STRIKEOUT:16,BOLD:32,REGULAR:64,USER_TYPO_METRICS:128,WWS:256,OBLIQUE:512},ao.prototype.macStyleValues={BOLD:1,ITALIC:2,UNDERLINE:4,OUTLINED:8,SHADOW:16,CONDENSED:32,EXTENDED:64},ao.prototype.usWidthClasses={ULTRA_CONDENSED:1,EXTRA_CONDENSED:2,CONDENSED:3,SEMI_CONDENSED:4,MEDIUM:5,SEMI_EXPANDED:6,EXPANDED:7,EXTRA_EXPANDED:8,ULTRA_EXPANDED:9},ao.prototype.usWeightClasses={THIN:100,EXTRA_LIGHT:200,LIGHT:300,NORMAL:400,MEDIUM:500,SEMI_BOLD:600,BOLD:700,EXTRA_BOLD:800,BLACK:900};var io=ao;var lo={make:function(){console.warn("Writing of hvar tables is not yet supported.")},parse:function(e,t){const n=new Ee.Parser(e,t),r=n.parseUShort(),s=n.parseUShort();return 1!==r&&console.warn(`Unsupported hvar table version ${r}.${s}`),{version:[r,s],itemVariationStore:n.parsePointer32((function(){return this.parseItemVariationStore()})),advanceWidth:n.parsePointer32((function(){return this.parseDeltaSetIndexMap()})),lsb:n.parsePointer32((function(){return this.parseDeltaSetIndexMap()})),rsb:n.parsePointer32((function(){return this.parseDeltaSetIndexMap()}))}}},uo=function(){return{coverage:this.parsePointer(Ce.coverage),attachPoints:this.parseList(Ce.pointer(Ce.uShortList))}},co=function(){var e=this.parseUShort();return G.argument(1===e||2===e||3===e,"Unsupported CaretValue table version."),1===e?{coordinate:this.parseShort()}:2===e?{pointindex:this.parseShort()}:3===e?{coordinate:this.parseShort()}:void 0},po=function(){return this.parseList(Ce.pointer(co))},ho=function(){return{coverage:this.parsePointer(Ce.coverage),ligGlyphs:this.parseList(Ce.pointer(po))}},fo=function(){return this.parseUShort(),this.parseList(Ce.pointer(Ce.coverage))};var go={parse:function(e,t){const n=new Ce(e,t=t||0),r=n.parseVersion(1);G.argument(1===r||1.2===r||1.3===r,"Unsupported GDEF table version.");var s={version:r,classDef:n.parsePointer(Ce.classDef),attachList:n.parsePointer(uo),ligCaretList:n.parsePointer(ho),markAttachClassDef:n.parsePointer(Ce.classDef)};return r>=1.2&&(s.markGlyphSets=n.parsePointer(fo)),s}},mo=new Array(10);mo[1]=function(){const e=this.offset+this.relativeOffset,t=this.parseUShort();return 1===t?{posFormat:1,coverage:this.parsePointer(Ce.coverage),value:this.parseValueRecord()}:2===t?{posFormat:2,coverage:this.parsePointer(Ce.coverage),values:this.parseValueRecordList()}:void G.assert(!1,"0x"+e.toString(16)+": GPOS lookup type 1 format must be 1 or 2.")},mo[2]=function(){const e=this.offset+this.relativeOffset,t=this.parseUShort();G.assert(1===t||2===t,"0x"+e.toString(16)+": GPOS lookup type 2 format must be 1 or 2.");const n=this.parsePointer(Ce.coverage),r=this.parseUShort(),s=this.parseUShort();if(1===t)return{posFormat:t,coverage:n,valueFormat1:r,valueFormat2:s,pairSets:this.parseList(Ce.pointer(Ce.list((function(){return{secondGlyph:this.parseUShort(),value1:this.parseValueRecord(r),value2:this.parseValueRecord(s)}}))))};if(2===t){const e=this.parsePointer(Ce.classDef),o=this.parsePointer(Ce.classDef),a=this.parseUShort(),i=this.parseUShort();return{posFormat:t,coverage:n,valueFormat1:r,valueFormat2:s,classDef1:e,classDef2:o,class1Count:a,class2Count:i,classRecords:this.parseList(a,Ce.list(i,(function(){return{value1:this.parseValueRecord(r),value2:this.parseValueRecord(s)}})))}}},mo[3]=function(){return{error:"GPOS Lookup 3 not supported"}},mo[4]=function(){return{error:"GPOS Lookup 4 not supported"}},mo[5]=function(){return{error:"GPOS Lookup 5 not supported"}},mo[6]=function(){return{error:"GPOS Lookup 6 not supported"}},mo[7]=function(){return{error:"GPOS Lookup 7 not supported"}},mo[8]=function(){return{error:"GPOS Lookup 8 not supported"}},mo[9]=function(){return{error:"GPOS Lookup 9 not supported"}};var yo=new Array(10);var vo={parse:function(e,t){const n=new Ce(e,t=t||0),r=n.parseVersion(1);return G.argument(1===r||1.1===r,"Unsupported GPOS table version "+r),1===r?{version:r,scripts:n.parseScriptList(),features:n.parseFeatureList(),lookups:n.parseLookupList(mo)}:{version:r,scripts:n.parseScriptList(),features:n.parseFeatureList(),lookups:n.parseLookupList(mo),variations:n.parseFeatureVariationsList()}},make:function(e){return new ie.Table("GPOS",[{name:"version",type:"ULONG",value:65536},{name:"scripts",type:"TABLE",value:new ie.ScriptList(e.scripts)},{name:"features",type:"TABLE",value:new ie.FeatureList(e.features)},{name:"lookups",type:"TABLE",value:new ie.LookupList(e.lookups,yo)}])}};var bo={parse:function(e,t){const n=new Ee.Parser(e,t),r=n.parseUShort();if(0===r)return function(e){const t={};e.skip("uShort");const n=e.parseUShort();G.argument(0===n,"Unsupported kern sub-table version."),e.skip("uShort",2);const r=e.parseUShort();e.skip("uShort",3);for(let n=0;n<r;n+=1){const n=e.parseUShort(),r=e.parseUShort(),s=e.parseShort();t[n+","+r]=s}return t}(n);if(1===r)return function(e){const t={};e.skip("uShort"),e.parseULong()>1&&console.warn("Only the first kern subtable is supported."),e.skip("uLong");const n=255&e.parseUShort();if(e.skip("uShort"),0===n){const n=e.parseUShort();e.skip("uShort",3);for(let r=0;r<n;r+=1){const n=e.parseUShort(),r=e.parseUShort(),s=e.parseShort();t[n+","+r]=s}}return t}(n);throw new Error("Unsupported kern table version ("+r+").")}};var So,xo={parse:function(e,t,n,r){const s=new Ee.Parser(e,t),o=r?s.parseUShort:s.parseULong,a=[];for(let e=0;e<n+1;e+=1){let e=o.call(s);r&&(e*=2),a.push(e)}return a}};function To(e,t){const n=[];let r=12;for(let s=0;s<t;s+=1){const t=Ee.getTag(e,r),s=Ee.getULong(e,r+4),o=Ee.getULong(e,r+8),a=Ee.getULong(e,r+12);n.push({tag:t,checksum:s,offset:o,length:a,compression:!1}),r+=16}return n}function ko(e,t){if("WOFF"===t.compression){const n=new Uint8Array(e.buffer,t.offset+2,t.compressedLength-2),r=new Uint8Array(t.length);if(w(n,r),r.byteLength!==t.length)throw new Error("Decompression error: "+t.tag+" decompressed length doesn't match recorded length");return{data:new DataView(r.buffer,0),offset:0}}return{data:e,offset:t.offset}}function Oo(e,t={}){let n,r;const s=new io({empty:!0});e.constructor!==ArrayBuffer&&(e=new Uint8Array(e).buffer);const o=new DataView(e,0);let a,i=[];const l=Ee.getTag(o,0);if(l===String.fromCharCode(0,1,0,0)||"true"===l||"typ1"===l)s.outlinesFormat="truetype",a=Ee.getUShort(o,4),i=To(o,a);else if("OTTO"===l)s.outlinesFormat="cff",a=Ee.getUShort(o,4),i=To(o,a);else{if("wOFF"!==l){if("wOF2"===l){throw new Error("WOFF2 require an external decompressor library, see examples at: https://github.com/opentypejs/opentype.js/issues/183#issuecomment-1147228025")}throw new Error("Unsupported OpenType signature "+l)}{const e=Ee.getTag(o,4);if(e===String.fromCharCode(0,1,0,0))s.outlinesFormat="truetype";else{if("OTTO"!==e)throw new Error("Unsupported OpenType flavor "+l);s.outlinesFormat="cff"}a=Ee.getUShort(o,12),i=function(e,t){const n=[];let r=44;for(let s=0;s<t;s+=1){const t=Ee.getTag(e,r),s=Ee.getULong(e,r+4),o=Ee.getULong(e,r+8),a=Ee.getULong(e,r+12);let i;i=o<a&&"WOFF",n.push({tag:t,offset:s,compression:i,compressedLength:o,length:a}),r+=20}return n}(o,a)}}let u,c,p,h,f,d,g,m,y,v,b,S,x,T,k,O,U,R;for(let e=0;e<a;e+=1){const t=i[e];let a;switch(t.tag){case"avar":g=t;break;case"cmap":a=ko(o,t),s.tables.cmap=Xe.parse(a.data,a.offset),s.encoding=new st(s.tables.cmap);break;case"cvt ":a=ko(o,t),R=new Ee.Parser(a.data,a.offset),s.tables.cvt=R.parseShortList(t.length/2);break;case"fvar":p=t;break;case"STAT":h=t;break;case"gvar":f=t;break;case"cvar":d=t;break;case"fpgm":a=ko(o,t),R=new Ee.Parser(a.data,a.offset),s.tables.fpgm=R.parseByteList(t.length);break;case"head":a=ko(o,t),s.tables.head=$t.parse(a.data,a.offset),s.unitsPerEm=s.tables.head.unitsPerEm,n=s.tables.head.indexToLocFormat;break;case"hhea":a=ko(o,t),s.tables.hhea=Yt.parse(a.data,a.offset),s.ascender=s.tables.hhea.ascender,s.descender=s.tables.hhea.descender,s.numberOfHMetrics=s.tables.hhea.numberOfHMetrics;break;case"HVAR":x=t;break;case"hmtx":S=t;break;case"ltag":a=ko(o,t),r=Zt.parse(a.data,a.offset);break;case"COLR":a=ko(o,t),s.tables.colr=an.parse(a.data,a.offset);break;case"CPAL":a=ko(o,t),s.tables.cpal=gt.parse(a.data,a.offset);break;case"maxp":a=ko(o,t),s.tables.maxp=Qt.parse(a.data,a.offset),s.numGlyphs=s.tables.maxp.numGlyphs;break;case"name":O=t;break;case"OS/2":a=ko(o,t),s.tables.os2=Jt.parse(a.data,a.offset);break;case"post":a=ko(o,t),s.tables.post=en.parse(a.data,a.offset),s.glyphNames=new at(s.tables.post);break;case"prep":a=ko(o,t),R=new Ee.Parser(a.data,a.offset),s.tables.prep=R.parseByteList(t.length);break;case"glyf":m=t;break;case"loca":k=t;break;case"CFF ":u=t;break;case"CFF2":c=t;break;case"kern":T=t;break;case"GDEF":y=t;break;case"GPOS":v=t;break;case"GSUB":b=t;break;case"meta":U=t;break;case"gasp":try{a=ko(o,t),s.tables.gasp=kn.parse(a.data,a.offset)}catch(e){console.warn("Failed to parse gasp table, skipping."),console.warn(e)}break;case"SVG ":a=ko(o,t),s.tables.svg=On.parse(a.data,a.offset)}}const w=ko(o,O);if(s.tables.name=$e.parse(w.data,w.offset,r),s.names=s.tables.name,m&&k){const e=0===n,r=ko(o,k),a=xo.parse(r.data,r.offset,s.numGlyphs,e),i=ko(o,m);s.glyphs=ar.parse(i.data,i.offset,a,s,t)}else if(u){const e=ko(o,u);Wt.parse(e.data,e.offset,s,t)}else{if(!c)throw new Error("Font doesn't contain TrueType, CFF or CFF2 outlines.");{const e=ko(o,c);Wt.parse(e.data,e.offset,s,t)}}const I=ko(o,S);if(Xt.parse(s,I.data,I.offset,s.numberOfHMetrics,s.numGlyphs,s.glyphs,t),it(s,t),T){const e=ko(o,T);s.kerningPairs=bo.parse(e.data,e.offset)}else s.kerningPairs={};if(y){const e=ko(o,y);s.tables.gdef=go.parse(e.data,e.offset)}if(v){const e=ko(o,v);s.tables.gpos=vo.parse(e.data,e.offset),s.position.init()}if(b){const e=ko(o,b);s.tables.gsub=sn.parse(e.data,e.offset)}if(p){const e=ko(o,p);s.tables.fvar=pn.parse(e.data,e.offset,s.names)}if(h){const e=ko(o,h);s.tables.stat=yn.parse(e.data,e.offset,s.tables.fvar)}if(f){p||console.warn("This font provides a gvar table, but no fvar table, which is required for variable fonts."),m||console.warn("This font provides a gvar table, but no glyf table. Glyph variation only works with TrueType outlines.");const e=ko(o,f);s.tables.gvar=Tn.parse(e.data,e.offset,s.tables.fvar,s.glyphs)}if(d){p||console.warn("This font provides a cvar table, but no fvar table, which is required for variable fonts."),s.tables.cvt||console.warn("This font provides a cvar table, but no cvt table which could be made variable."),m||console.warn("This font provides a gvar table, but no glyf table. Glyph variation only works with TrueType outlines.");const e=ko(o,d);s.tables.cvar=xn.parse(e.data,e.offset,s.tables.fvar,s.tables.cvt||[])}if(g){p||console.warn("This font provides an avar table, but no fvar table, which is required for variable fonts.");const e=ko(o,g);s.tables.avar=Sn.parse(e.data,e.offset,s.tables.fvar)}if(x){p||console.warn("This font provides an HVAR table, but no fvar table, which is required for variable fonts."),S||console.warn("This font provides an HVAR table, but no hmtx table to vary.");const e=ko(o,x);s.tables.hvar=lo.parse(e.data,e.offset,s.tables.fvar)}if(U){const e=ko(o,U);s.tables.meta=on.parse(e.data,e.offset),s.metas=s.tables.meta}return s.palettes=new Wn(s),s}function Uo(){console.error("DEPRECATED! migrate to: opentype.parse(buffer, opt) See: https://github.com/opentypejs/opentype.js/issues/675")}function Ro(){console.error('DEPRECATED! migrate to: opentype.parse(require("fs").readFileSync(url), opt)')}return So=s,((s,o,a,i)=>{if(o&&"object"==typeof o||"function"==typeof o)for(let l of n(o))r.call(s,l)||l===a||e(s,l,{get:()=>o[l],enumerable:!(i=t(o,l))||i.enumerable});return s})(e({},"__esModule",{value:!0}),So)})();!function(e,t){"function"==typeof define&&define.amd?define(t):"object"==typeof module&&module.exports?module.exports=t():e.opentype=t()}("undefined"!=typeof self?self:this,(()=>({...opentype,default:opentype})));
/////////////////////end of opentype.js///////////////////////

  var parameters = PluginManager.parameters('OpenTypePixelFont');
  var openfontList = [];
  var findex = 0;
  window.openFontLoading = 0;
  window.openfontList = openfontList;
  var paramFontList = JSON.parse(parameters['fontList']);

  //fix hash for MZ
  Utils.encodeURI = function (str) {
    return encodeURIComponent(str).replace(/%2F/g, "/").replace(/%23/g, "#");
  };

  //Try to find pixel size of the glyph, this does not always work
  //You need to try it on different characters and use the smallest value.
  function findMinSeg(points) {
    var minx, minx2, miny, miny2;
    minx = minx2 = miny = miny2 = 123456789;
    for (var i = 1; i < points.length; i++) {
      var dx = Math.abs(points[i].x - points[i - 1].x);
      var dy = Math.abs(points[i].y - points[i - 1].y);
      if (dx && dx < minx) {
        minx2 = minx;
        minx = dx;
      }
      if (dy && dy < miny) {
        miny2 = miny;
        miny = dy;
      }
    }
    var xseg, yseg;
    if (Math.round(minx / minx2) == 1) {
      xseg = (minx + minx2) / 2;
    } else {
      xseg = minx;
    }
    if (Math.round(miny / miny2) == 1) {
      yseg = (miny + miny2) / 2;
    } else {
      yseg = miny;
    }
    return Math.min(xseg, yseg);
  }

  for (var i = 0; i < paramFontList.length; i++) {
    (function () {
      var fobj = JSON.parse(paramFontList[i]);
      var ff = fobj.file;
      var fa = fobj.family.toLowerCase();
      var fs = eval(fobj.snapToPixel);
      var bs;
      eval('bs=' + fobj.baseSize);
      var ur = fobj['unicode-range'];
      if (!fa || !ff) return;
      var fp = './fonts/' + ff;
      var index = findex++;
      //openfontList[fa] = fp;
      openFontLoading++;
      fetch(fp).then(res => res.arrayBuffer()).then(data => {
        var fnt = opentype.parse(data);
        fnt.snap = fs;
        fnt.shouldSquishText = ff.indexOf('#squish') > 0;
        fnt.baseSize = bs;
        fnt.unicodeRange = parseUnicodRange(ur.split(','));
        fnt.familyName = fa;
        var seg = 99999999;
        var notPixel = false;
        const cmds = ['M', 'L', 'Z'];
        var height = fnt.unitsPerEm || 1e3;
        for (var ind in fnt.glyphs.glyphs) {
          var g = fnt.glyphs.glyphs[ind];
          if (!notPixel && g.unicode >= 65 && g.unicode <= 122) {
            var path = g.path;
            if (path.commands.some((c) => cmds.indexOf(c.type) < 0)) {
              notPixel = true;
            } else if (g.points.length > 2) {
              var sg = findMinSeg(g.points);
              if (sg < seg) seg = sg;
            }
          }
        }
        fnt._baseSize = notPixel ? 1 : Math.round(height / seg);
        openfontList[index] = fnt;
        openFontLoading--;
      }, function () { openFontLoading--; console.log('Loading ' + fp + ' failed') });
    })();
  }

  //wait for font loading
  (function (alias) {
    Scene_Boot.prototype.isReady = function () {
      if (openFontLoading) return false;
      return alias.call(this);
    }
  })(Scene_Boot.prototype.isReady);

  //simulate italic style, y is the baseline position
  function simItalic(path, y) {
    const layers = path._layers;
    if (layers && layers.length) {
      for (let l = 0; l < layers.length; l++) {
        simItalic(layers[l]);
      }
      return;
    }
    var angle = 0.2;
    for (let i = 0; i < path.commands.length; i++) {
      var cmd = path.commands[i];
      if (cmd.type !== 'Z') {
        cmd.x += (y - cmd.y) * angle;
      }
      if (cmd.type === 'Q' || cmd.type === 'C') {
        cmd.x1 += (y - cmd.y1) * angle;
      }
      if (cmd.type === 'C') {
        cmd.x2 += (y - cmd.y2) * angle;
      }
    }
  }

  function doScale(path, x, scale) {
    const layers = path._layers;
    if (layers && layers.length) {
      for (let l = 0; l < layers.length; l++) {
        doScale(layers[l], x, scale);
      }
      return;
    }
    for (let i = 0; i < path.commands.length; i++) {
      var cmd = path.commands[i];
      if (cmd.type !== 'Z') {
        cmd.x = (cmd.x - x) * scale + x;
      }
      if (cmd.type === 'Q' || cmd.type === 'C') {
        cmd.x1 = (cmd.x1 - x) * scale + x;
      }
      if (cmd.type === 'C') {
        cmd.x2 = (cmd.x2 - x) * scale + x;
      }
    }
  }

  function doSnap(path) {//path.commands.forEach(c=>console.log(c.x, c.y));console.log('--------------------')
    const layers = path._layers;
    if (layers && layers.length) {
      for (let l = 0; l < layers.length; l++) {
        doSnap(layers[l]);
      }
      return;
    }
    var testPixelFont = true;
    var dx = undefined, dy = undefined;
    function snap(cmd, c, dc) {
      var cmdc = cmd[c];
      cmd[c] = Math.round(cmd[c] + (dc || 0));
      if (undefined === dc) dc = cmd[c] - cmdc;
      return dc;
    }
    // Round a value to the nearest "step".
    for (let i = 0; i < path.commands.length; i++) {
      var cmd = path.commands[i];
      if (cmd.type !== 'Z') {
        dx = snap(cmd, 'x', dx);
        dy = snap(cmd, 'y', dy);
      }
      if (cmd.type === 'Q' || cmd.type === 'C') {
        dx = snap(cmd, 'x1', dx);
        dy = snap(cmd, 'y1', dy);
        testPixelFont = false;
      }
      if (cmd.type === 'C') {
        dx = snap(cmd, 'x2', dx);
        dy = snap(cmd, 'y2', dy);
      }
    }
    //path.commands.forEach(c=>console.log(c));console.log('--------------------')
    if (!testPixelFont) return;

    //merge points
    var p = null, head = null;
    for (let i = 0; i < path.commands.length; i++) {
      var cmd = path.commands[i];
      if (cmd.type === 'M') {
        if (head) head.prev = p;
        head = p = cmd;
      } else if (cmd.type === 'L') {
        p.next = cmd;
        cmd.prev = p;
        p = cmd;
      }
    }
    if (head) head.prev = p;
    for (let i = 0; i < path.commands.length; i++) {
      var cmd = path.commands[i];
      if (cmd.type === 'L' || cmd.type === 'M') {
        if ((cmd.x == cmd.prev.x && cmd.prev.x == cmd.prev.prev.x) || cmd.y == cmd.prev.y && cmd.prev.y == cmd.prev.prev.y) {
          cmd.prev.deleted = true;
          if (cmd.prev.type === 'M') cmd.type = 'M';
          cmd.prev.prev.next = cmd;
          cmd.prev = cmd.prev.prev;
        }
      }
    }
    path.commands = path.commands.filter(function (c) {
      return !c.deleted;
    });
  }

  var regUnicode = /^u\+(?:([0-9a-f]?[0-9a-f?]{1,5})|([0-9a-f]{1,6})-([0-9a-f]{1,6}))?$/i;

  //merge unicode ranges in an array
  function mergeRanges(ranges) {
    if (ranges.length === 0) return [];
    ranges.sort((a, b) => a[0] - b[0]);
    const merged = [ranges[0]];
    for (let i = 1; i < ranges.length; i++) {
      const current = ranges[i];
      const lastMerged = merged[merged.length - 1];
      if (current[0] <= lastMerged[1] + 1) {
        lastMerged[1] = Math.max(lastMerged[1], current[1]);
      } else {
        merged.push(current);
      }
    }
    return merged;
  }

  function parseUnicodRange(arr) {
    var result = [];

    var invalidstr = "Invalid unicode range:" + range;
    for (var range of arr) {
      range = range.trim()
      if (!regUnicode.test(range)) {
        throw invalidstr;
      }

      var [, single, start, end] = range.match(regUnicode);

      // Single unicode-range (e.g. U+20, U+3F U+30??)
      if (single) {
        if (/\?[^?]+$/.test(single)) {
          throw invalidstr;
        }
        if (single.includes('?')) {
          var s = parseInt(single.replace(/\?/g, '0'), 16);
          var e = parseInt(single.replace(/\?/g, 'F'), 16);
          result.push([s, e]);
        } else {
          var s = parseInt(single, 16);
          result.push([s, s]);
        }
      }

      // Interval unicode-range (e.g. U+30-39)
      if (start && end) {
        var s = parseInt(start, 16);
        var e = parseInt(end, 16);
        result.push([s, e]);
      }
    }

    return mergeRanges(result);
  }

  var $$font = {};//cache the results
  var $$$p = document.createElement('p');
  function getFont(font) {
    var f = $$font[font];
    if (!f) {
      $$$p.style.font = font;
      f = {};
      f.fontSize = $$$p.style.fontSize;
      f.fontFamily = $$$p.style.fontFamily.replace(/['"]/g, '').toLowerCase() || 'gamefont';
      f.fontStyle = $$$p.style.fontStyle;
      f.fontWeight = $$$p.style.fontWeight;
      f.size = parseInt(f.fontSize);
      var fmls = f.fontFamily.split(',');
      var flist = [];
      for (var i = 0; i < fmls.length; i++) {
        fmls[i] = fmls[i].trim();
        var ff = openfontList.filter(function (v) { return v.familyName == fmls[i]; });
        if (ff) flist.push(...ff);
      }
      f.fontFamilies = fmls;
      f.unicodeMap = [];
      f.flist = flist;
      f.isOpenFont = flist.length != 0;
      $$font[font] = f;
    }
    return f;
  }

  CanvasRenderingContext2D.prototype._groupTextForDrawing = function (t, f, w) {
    var groups = [];
    var g = { text: '' };
    for (var s = 0; s < t.length; s++) {
      var code = t.charCodeAt(s);
      var fnt = null;
      loop: for (var i = 0; i < f.flist.length; i++) {
        var ur = f.flist[i].unicodeRange;
        for (var u of ur) {
          if (u[0] <= code && u[1] >= code) {
            fnt = f.flist[i];
            break loop;
          }
        }
      }
      if (fnt !== g.font) {
        g = { text: t[s], font: fnt };
        groups.push(g);
      } else {
        g.text += t[s];
      }
    }
    var width = 0;
    groups.forEach(g => {
      g.width = this._measureText(g, f);
      width += g.width;
    });
    var advanceWidth = 0;
    var scale = width > w ? w / width : 1;
    groups.forEach(g => {
      g.scale = advanceScale(g, f, g.width * scale);
      advanceWidth += g.scale * g.width;
    });
    groups.width = width;
    groups.advanceWidth = advanceWidth || width;
    return groups;
  }

  function findOptimalSize(font, size) {
    var baseSize = font.baseSize;
    if (typeof baseSize == 'number') {
      if (baseSize > 1 && size > baseSize) {
        return Math.round(size / baseSize) * baseSize;
      } else if (size <= baseSize) {
        return baseSize;
      }
    } else if (typeof baseSize == 'object') {
      return baseSize[size] || size;
    } else if (typeof baseSize == 'function') {
      return baseSize(font, size);
    }
    return size;
  }

  function advanceScale(g, f, w) {
    if (!g.font) return w / g.width;
    if (!g.font.shouldSquishText) return 1;
    var scale = 1;
    var osize = findOptimalSize(g.font, f.size);
    if (g.width > w) {
      var scalew = w / g.width;
      for (var s = f.size - 1; s > 5; s--) {
        var ns = findOptimalSize(g.font, s);
        scale = ns / osize;
        if (scale <= scalew) {
          break;
        }
      }
    }
    return scale;
  }

  var $mt = CanvasRenderingContext2D.prototype.measureText;
  CanvasRenderingContext2D.prototype._measureText = function (g, f) {
    var t = g.text, of = g.font;
    //var path = of.getPath(t, 0, 0, f.size, this.options||of.defaultRenderOptions);
    //should italic be used here?
    //if(f.fontStyle.contains('italic')){
    //  simItalic(path,y);
    //}
    //if(of.snap){
    //  doSnap(path);
    //}
    //var box = path.getBoundingBox();
    if (!of) {
      return $mt.call(this, t).width;
    }
    return of.getAdvanceWidth(t, findOptimalSize(of, f.size), this.options || of.defaultRenderOptions)
  };
  CanvasRenderingContext2D.prototype.measureText = function (t, w) {
    t += '';
    var f = getFont(this.font);
    if (f.isOpenFont) {
      return this._groupTextForDrawing(t, f, w);
    } else {
      var size = $mt.call(this, t);
      size.advanceWidth = size.width * (w < size.width ? w / size.width : 1);
      return size;
    }
  };

  const cmdM = { type: 'M', x: -12345, y: -12345 };
  const cmdL = { type: 'L', x: -12345, y: -12345 };
  CanvasRenderingContext2D.prototype._pathText = function (g, f, x, y, w) {
    var t = g.text;
    var of = g.font;
    var osize = findOptimalSize(of, f.size);
    var path = of.getPath(t, x, y + (osize - f.size) / 2, osize, this.options || of.defaultRenderOptions);
    if (f.fontStyle.contains('italic')) {
      simItalic(path, y);
    }
    if (g.scale < 1 && of.shouldSquishText) doScale(path, x, g.scale);
    if (of.snap) {
      doSnap(path);
    }
    if (osize == of._baseSize || g.scale * osize == of._baseSize) {
      for (let i = 0; i < 17; i++) { //dummy commands, fix gpu glitch 
        path.commands.push(cmdM, cmdL, cmdL, cmdL);
      }
    }
    return path;
  };

  var $ft = CanvasRenderingContext2D.prototype.fillText;
  CanvasRenderingContext2D.prototype._fillText = function (g, f, x, y, w) {
    if (g.font) {
      var path = this._pathText(g, f, x, y, w);
      path.fill = true; path.stroke = false;
      path.draw(this);
    } else {
      $ft.call(this, g.text, x, y, w);
    }
  };
  CanvasRenderingContext2D.prototype.fillText = function (t, x, y, w) {
    t += '';
    var f = getFont(this.font);
    if (f.isOpenFont) {
      var groups = this.measureText(t, w);
      if (this.textAlign == 'center') {
        x -= groups.advanceWidth / 2;
      } else if (this.textAlign == 'right') {
        x -= groups.advanceWidth;
      }
      this.save();
      this.textAlign = 'left';
      this.imageSmoothingEnabled = false;
      for (g of groups) {
        var adw = g.width * g.scale;
        this._fillText(g, f, x, y, adw);
        x += adw;
      }
      this.restore();
    } else {
      return $ft.call(this, t, x, y, w);
    }
  };

  var $st = CanvasRenderingContext2D.prototype.strokeText;
  CanvasRenderingContext2D.prototype._strokeText = function (g, f, x, y, w) {
    if (g.font) {
      var path = this._pathText(g, f, x, y, w);
      path.fill = false; path.stroke = true; path.strokeWidth = Math.round(this.lineWidth / 2) * 2;
      path.draw(this);
    } else {
      $st.call(this, g.text, x, y, w);
    }
  };
  CanvasRenderingContext2D.prototype.strokeText = function (t, x, y, w) {
    t += '';
    var f = getFont(this.font);
    if (f.isOpenFont) {
      var groups = this.measureText(t, w);
      if (this.textAlign == 'center') {
        x -= groups.advanceWidth / 2;
      } else if (this.textAlign == 'right') {
        x -= groups.advanceWidth;
      }
      this.save();
      this.textAlign = 'left';
      this.lineCap = 'square';
      this.lineJoin = 'miter';
      this.imageSmoothingEnabled = false;
      for (g of groups) {
        var adw = g.width * g.scale;
        this._strokeText(g, f, x, y, adw);
        x += adw;
      }
      this.restore();
    } else {
      return $st.call(this, t, x, y, w);
    }
  };
})();